package com.kontakt.sdk.android.cloud.api.executor.firmwares;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.FirmwaresApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.FirmwaresService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link FirmwaresApi}. Use this class if you want to schedule
 * firmware update through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.firmwares().scheduleUpdate()
 *      .forDevices("4DgT", "8JkU")
 *      .withVersion("4.0")
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that devices and version must be specified so invocations of {@code forDevices}
 * and {@code withVersion} methods are mandatory. Otherwise an exception will be thrown.
 */
public class ScheduleFirmwareUpdateRequestExecutor extends RequestExecutor<String> {

  private final FirmwaresService firmwaresService;

  private String[] deviceUniqueIds;
  private String name;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param firmwaresService the firmwares API service.
   */
  public ScheduleFirmwareUpdateRequestExecutor(final FirmwaresService firmwaresService) {
    this.firmwaresService = firmwaresService;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param deviceUniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public ScheduleFirmwareUpdateRequestExecutor forDevices(final String... deviceUniqueIds) {
    this.deviceUniqueIds = SDKPreconditions.checkNotNull(deviceUniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param deviceUniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public ScheduleFirmwareUpdateRequestExecutor forDevices(final List<String> deviceUniqueIds) {
    SDKPreconditions.checkNotNull(deviceUniqueIds, "IDs cannot be null");
    final int size = deviceUniqueIds.size();
    this.deviceUniqueIds = deviceUniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies the version name. The method invocation is obligatory while using this request executor.
   *
   * @param name the version name.
   * @return this request executor.
   */
  public ScheduleFirmwareUpdateRequestExecutor withVersion(final String name) {
    this.name = SDKPreconditions.checkNotNull(name, "name cannot be null");
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return firmwaresService.scheduleFirmwareUpdate(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(deviceUniqueIds != null, "specify devices");
    SDKPreconditions.checkState(name != null, "specify firmware version name");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(deviceUniqueIds, ","));
    params.put(CloudConstants.Firmwares.NAME_PARAMETER, name);
    return params;
  }
}
