package com.kontakt.sdk.android.cloud.api.executor.activities;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActivitiesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActivitiesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Activity;
import com.kontakt.sdk.android.common.model.ActivityContext;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link ActivitiesApi}. Use this class if you want to update
 * activities through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   // define updated activity object
 *   kontaktCloud.activities().update(ID)
 *      .with(updatedActivity)
 *      .execute();
 *   </code>
 * </pre>
 */
public class UpdateActivityRequestExecutor extends RequestExecutor<String> {

  private final ActivitiesService activitiesService;
  private final UUID id;
  private Activity activity;

  /**
   * Constructs request executor initialized with corresponding service class and activity unique ID.
   *
   * @param activitiesService the activities API service.
   * @param id activity unique identifier.
   */
  public UpdateActivityRequestExecutor(final ActivitiesService activitiesService, final UUID id) {
    this.activitiesService = activitiesService;
    this.id = id;
  }

  /**
   * Specifies activity data to update. The method invocation is obligatory while using this request executor.
   * a
   *
   * @param activity updated activity data.
   * @return this request executor.
   */
  public UpdateActivityRequestExecutor with(final Activity activity) {
    SDKPreconditions.checkNotNull(activity, "Activity cannot be null");
    this.activity = activity;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return activitiesService.update(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(id != null, "Cannot update activity - specify activity ID");
    SDKPreconditions.checkState(activity != null, "Cannot update activity - specify updated activity");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    addActivityParams(params);
    addActivityContextParams(params);
    return params;
  }

  private void addActivityParams(Map<String, String> params) {
    params.put(CloudConstants.Activities.ACTIVITY_ID_PARAMETER, id.toString());
    if (activity.getName() != null) {
      params.put(CloudConstants.Activities.NAME_PARAMETER, activity.getName());
    }
    if (activity.getType() != null) {
      params.put(CloudConstants.Activities.TYPE_PARAMETER, activity.getType().name());
    }
    List<UUID> triggersIds = activity.getTriggerIds();
    if (triggersIds != null && !triggersIds.isEmpty()) {
      params.put(CloudConstants.Activities.TRIGGER_ID_PARAMETER, StringUtils.join(triggersIds, ","));
    }
  }

  private void addActivityContextParams(Map<String, String> params) {
    ActivityContext context = activity.getContext();
    if (context != null) {
      if (context.getHttpMethod() != null) {
        params.put(CloudConstants.Activities.CONTEXT_HTTP_METHOD, activity.getContext().getHttpMethod().name());
      }
      if (context.getUrl() != null) {
        params.put(CloudConstants.Activities.CONTEXT_URL, activity.getContext().getUrl());
      }
      Map<String, String> headers = activity.getContext().getHeaders();
      Map<String, String> parameters = activity.getContext().getParameters();
      if (headers != null) {
        for (Map.Entry<String, String> entry : headers.entrySet()) {
          params.put(CloudConstants.Activities.CONTEXT_HEADERS + entry.getKey(), entry.getValue());
        }
      }
      if (parameters != null) {
        for (Map.Entry<String, String> entry : parameters.entrySet()) {
          params.put(CloudConstants.Activities.CONTEXT_HEADERS + entry.getKey(), entry.getValue());
        }
      }
    }
  }
}
