package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.receivers.CreateReceiverRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.receivers.DeleteReceiverRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.receivers.ReceiversRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.receivers.UpdateReceiverRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ReceiversService;
import com.kontakt.sdk.android.common.model.Receiver;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * An abstract representation of Receiver API endpoint. Provides executors that can be
 * used to handle all receiver-related requests.
 */
public class ReceiversApi {

  private final ReceiversService receiversService;

  /**
   * Constructs Receiver API endpoint initialized with corresponding service class.
   * @param receiversService the receivers API service.
   */
  public ReceiversApi(ReceiversService receiversService) {
    this.receiversService = receiversService;
  }

  /**
   * Provides executor that can be used to obtain receivers.
   *
   * @return the request executor.
   */
  public ReceiversRequestExecutor fetch() {
    return new ReceiversRequestExecutor(receiversService);
  }

  /**
   * Provides executor that can be used to create new receiver.
   *
   * @param receiver the receiver.
   * @return the request executor.
   */
  public CreateReceiverRequestExecutor create(final Receiver receiver) {
    checkNotNull(receiver, "receiver cannot be null");
    return new CreateReceiverRequestExecutor(receiversService, receiver);
  }

  /**
   * Provides executor that can be used to update a receiver.
   *
   * @param id the receiver's identifier
   * @return the request executor.
   */
  public UpdateReceiverRequestExecutor update(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new UpdateReceiverRequestExecutor(receiversService, id);
  }

  /**
   * Provides executor that can be used to delete a receiver.
   *
   * @param id the receiver's identifier.
   * @return the request executor.
   */
  public DeleteReceiverRequestExecutor delete(final UUID id) {
    return new DeleteReceiverRequestExecutor(receiversService, id);
  }
}
