package com.kontakt.sdk.android.ble.spec;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;

/**
 * Telemetry provides meta data specific for particular Eddystone device.
 */
public class Telemetry implements Parcelable {

  /**
   * The parcelable constant CREATOR.
   */
  public static final Creator<Telemetry> CREATOR = new Creator<Telemetry>() {
    @Override
    public Telemetry createFromParcel(Parcel source) {
      Bundle bundle = source.readBundle();

      return new Telemetry.Builder()
          .version(bundle.getInt(Constants.TLM.VERSION))
          .pduCount(bundle.getInt(Constants.TLM.PDU_COUNT))
          .timeSincePowerUp(bundle.getInt(Constants.TLM.TIME_SINCE_POWER_UP))
          .batteryVoltage(bundle.getDouble(Constants.TLM.BATTERY_VOLTAGE))
          .temperature(bundle.getDouble(Constants.TLM.TEMPERATURE))
          .build();
    }

    @Override
    public Telemetry[] newArray(int size) {
      return new Telemetry[size];
    }
  };

  private final double batteryVoltage;
  private final double temperature;
  @SerializedName("advertisementCount") private final int pduCount;
  @SerializedName("uptime") private final int timeSincePowerUp;
  private final int version;
  private final HashCodeBuilder hashCodeBuilder;

  /**
   * Instantiates a new Telemetry.
   *
   * @param builder the builder
   */
  public Telemetry(Builder builder) {
    this.batteryVoltage = builder.batteryVoltage;
    this.temperature = builder.temperature;
    this.pduCount = builder.pduCount;
    this.timeSincePowerUp = builder.timeSincePowerUp;
    this.version = builder.version;
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  /**
   * Provides battery voltage [mV].
   *
   * @return the battery voltage
   */
  public double getBatteryVoltage() {
    return batteryVoltage;
  }

  /**
   * Provides temperature (Celsius degrees).
   *
   * @return the temperature
   */
  public double getTemperature() {
    return temperature;
  }

  /**
   * Provides pdu count since last reboot.
   *
   * @return the pdu count
   */
  public int getPduCount() {
    return pduCount;
  }

  /**
   * Provides time since power up.
   *
   * @return the time since power up
   */
  public int getTimeSincePowerUp() {
    return timeSincePowerUp;
  }

  /**
   * Provides version.
   *
   * @return the version
   */
  public int getVersion() {
    return version;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof Telemetry)) {
      return false;
    }

    Telemetry telemetry = (Telemetry) o;

    return batteryVoltage == telemetry.batteryVoltage &&
        pduCount == telemetry.pduCount &&
        temperature == telemetry.temperature &&
        timeSincePowerUp == telemetry.timeSincePowerUp &&
        version == telemetry.version;
  }

  @Override
  public int hashCode() {
    return hashCodeBuilder.append(batteryVoltage).append(temperature).append(pduCount).append(timeSincePowerUp).append(version).build();
  }

  @Override
  public String toString() {
    return "Telemetry{" +
        "batteryVoltage=" + batteryVoltage +
        ", temperature=" + temperature +
        ", pduCount=" + pduCount +
        ", timeSincePowerUp=" + timeSincePowerUp +
        ", version=" + version +
        '}';
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    Bundle bundle = new Bundle();
    bundle.putDouble(Constants.TLM.BATTERY_VOLTAGE, batteryVoltage);
    bundle.putDouble(Constants.TLM.TEMPERATURE, temperature);
    bundle.putInt(Constants.TLM.PDU_COUNT, pduCount);
    bundle.putInt(Constants.TLM.TIME_SINCE_POWER_UP, timeSincePowerUp);
    bundle.putInt(Constants.TLM.VERSION, version);

    dest.writeBundle(bundle);
  }

  /**
   * Telemetry Builder.
   */
  public static class Builder {
    double batteryVoltage;
    double temperature;
    int pduCount;
    int timeSincePowerUp;
    int version;

    /**
     * Sets battery voltage.
     *
     * @param batteryVoltage the battery voltage
     * @return the battery voltage
     */
    public Builder batteryVoltage(double batteryVoltage) {
      this.batteryVoltage = batteryVoltage;
      return this;
    }

    /**
     * Sets temperature.
     *
     * @param temperature the temperature
     * @return the temperature
     */
    public Builder temperature(double temperature) {
      this.temperature = temperature;
      return this;
    }

    /**
     * Sets pdu count.
     *
     * @param pduCount the pdu count
     * @return the pdu count
     */
    public Builder pduCount(int pduCount) {
      this.pduCount = pduCount;
      return this;
    }

    /**
     * Sets time since power up.
     *
     * @param timeSincePowerUp the time since power up
     * @return the time since power up
     */
    public Builder timeSincePowerUp(int timeSincePowerUp) {
      this.timeSincePowerUp = timeSincePowerUp;
      return this;
    }

    /**
     * Sets version.
     *
     * @param version the version
     * @return the version
     */
    public Builder version(int version) {
      this.version = version;
      return this;
    }

    /**
     * Build telemetry.
     *
     * @return the telemetry
     */
    public Telemetry build() {
      return new Telemetry(this);
    }
  }
}
