package com.kontakt.sdk.android.ble.service;

import android.app.Service;
import android.content.Intent;
import android.os.Binder;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.ble.monitoring.IEventCollector;
import com.kontakt.sdk.android.common.interfaces.SDKSupplier;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.Closeables;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Proximity service executes requests scheduled by {@link InternalProximityManager}.
 */
public class ProximityService extends Service {

  static final String TAG = ProximityService.class.getSimpleName();

  /**
   * Message code informing backing service to initialize scan.
   */
  public static final int MESSAGE_INITIALIZE_SCAN = 1;

  /**
   * Message code informing backing service that BLE devices scan has restarted.
   */
  public static final int MESSAGE_RESTART_SCAN = 2;

  /**
   * Message code informing backing service to finish scan.
   */
  public static final int MESSAGE_FINISH_SCAN = 3;

  /**
   * Message code informing about attaching monitoring listener.
   * Attaching listeners should take place while {@link InternalProximityManager}
   * is scanning.
   */
  public static final int MESSAGE_ATTACH_MONITORING_LISTENER = 4;

  /**
   * Message code informing about detaching monitoring listener.
   */
  public static final int MESSAGE_DETACH_MONITORING_LISTENER = 5;

  /**
   * Message code informing service that scan has finished.
   */
  public static final int MESSAGE_WORK_FINISHED = 6;

  /**
   * The response message notifying that {@link InternalProximityManager}'s request was handled successfully.
   */
  public static final int MESSAGE_SERVICE_RESPONSE_OK = 200;

  private final ServiceScanConfiguration configuration = new ServiceScanConfiguration();
  private Messenger serviceMessenger;
  private ServiceBinder serviceBinder;
  private Handler messagingHandler;
  private ScanCompat scanCompat;

  @Override
  public void onCreate() {
    super.onCreate();
    messagingHandler = new MessagingHandler(this);
    serviceMessenger = new Messenger(messagingHandler);
    serviceBinder = new ServiceBinder(serviceMessenger);
  }

  @Override
  public int onStartCommand(Intent intent, int flags, int startId) {
    super.onStartCommand(intent, flags, startId);
    return START_NOT_STICKY;
  }

  @Override
  public IBinder onBind(Intent intent) {
    return serviceBinder;
  }

  @Override
  public void onDestroy() {
    super.onDestroy();
    configuration.clear();
    messagingHandler = null;
    serviceMessenger = null;
  }

  public final Handler getMessagingHandler() {
    return messagingHandler;
  }

  ServiceScanConfiguration getConfiguration() {
    return configuration;
  }

  void onStartScan(final Bundle bundle) {
    ScanContext scanContext = bundle.getScanContext();
    FutureShufflesCache shufflesCache = bundle.getShufflesCache();
    IEventCollector eventCollector = bundle.getEventCollector  ();

    createScanCompat(scanContext);

    onStopScan();

    ScanConfiguration previousConfiguration = configuration.getScanConfiguration();
    ScanConfiguration newConfiguration = scanCompat.createScanConfiguration(scanContext, shufflesCache, eventCollector);
    Closeables.closeQuietly(previousConfiguration);

    ForceScanScheduler forceScanScheduler = scanCompat.createForceScanScheduler(newConfiguration);
    ScanController scanController = scanCompat.createScanController(newConfiguration, forceScanScheduler);
    ServiceScanConfiguration.Item item = new ServiceScanConfiguration.Item(newConfiguration, forceScanScheduler, scanController);

    configuration.add(item);
    configuration.getScanController().start();

    updateState(State.SCANNING);
  }

  void onStopScan() {
    if (isScanning()) {
      createScanCompat();
      configuration.getScanController().stop();
      configuration.getForceScanScheduler().stop();
      updateState(State.IDLE);
    } else {
      Logger.d(TAG + ": Stop Ranging method requested but BeaconService is not is Ranging state.");
    }
  }

  void onAttachListener(final InternalProximityListener proximityListener) {
    configuration.addListener(proximityListener);
  }

  void onDetachListener(final InternalProximityListener proximityListener) {
    configuration.removeListener(proximityListener);
  }

  protected boolean isScanning() {
    return configuration.getState() == State.SCANNING;
  }

  protected void updateState(final State newState) {
    configuration.updateState(newState);
  }

  void onCleanUp() {
    final ServiceScanConfiguration.Item item = configuration.remove();
    Closeables.closeQuietly(item.scanConfiguration);

    ForceScanScheduler forceScanScheduler = item.forceScanScheduler;
    if (forceScanScheduler != null) {
      forceScanScheduler.finish();
    }

    item.scanController.stop();
  }

  private void createScanCompat() {
    if (scanCompat == null) {
      scanCompat = ScanCompatFactory.createScanCompat();
    }
  }

  private void createScanCompat(ScanContext scanContext){
    if(scanCompat == null){
      scanCompat = ScanCompatFactory.createScanCompat(scanContext.getBLEScannerTypeProvider().provideIntenseScannerEnabled());
    }
  }

  private static final class ServiceBinder extends Binder implements SDKSupplier<Messenger> {
    private final Messenger serviceMessenger;

    ServiceBinder(Messenger serviceMessenger) {
      this.serviceMessenger = serviceMessenger;
    }

    @Override
    public Messenger get() {
      return serviceMessenger;
    }
  }

  private static final class MessagingHandler extends Handler {

    private final ProximityService service;

    MessagingHandler(ProximityService abstractBluetoothDeviceService) {
      super();
      service = abstractBluetoothDeviceService;
    }

    @SuppressWarnings("CatchMayIgnoreException")
    @Override
    public void handleMessage(Message msg) {
      try {
        Logger.d(TAG + " Message received.");
        switch (msg.what) {
          case MESSAGE_INITIALIZE_SCAN:
            Logger.d(TAG + " MESSAGE_INITIALIZE_SCAN");
            service.onStartScan((Bundle) msg.obj);
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;

          case MESSAGE_FINISH_SCAN:
            Logger.d(TAG + " MESSAGE_FINISH_SCAN");
            service.onStopScan();
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;

          case MESSAGE_RESTART_SCAN:
            Logger.d(TAG + " MESSAGE_RESTART_SCAN");
            service.onStopScan();
            service.onStartScan((Bundle) msg.obj);
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;

          case MESSAGE_WORK_FINISHED:
            Logger.d(TAG + " MESSAGE_WORK_FINISHED");
            service.onCleanUp();
            service.stopSelf();
            msg.replyTo.send(Message.obtain(null, msg.what, MESSAGE_SERVICE_RESPONSE_OK, 0));
            break;

          case MESSAGE_ATTACH_MONITORING_LISTENER:
            Logger.d(TAG + " MESSAGE_ATTACH_MONITORING_LISTENER");
            service.onAttachListener((InternalProximityListener) msg.obj);
            break;

          case MESSAGE_DETACH_MONITORING_LISTENER:
            Logger.d(TAG + " MESSAGE_DETACH_MONITORING_LISTENER");
            service.onDetachListener((InternalProximityListener) msg.obj);
            break;

          default:
            throw new IllegalStateException("Unsupported message code: " + msg.what);
        }
      } catch (Exception ignored) {
        Logger.e(TAG + " " + ignored.getMessage());
      }
    }
  }

  public static final class Bundle {

    private final ScanContext scanContext;
    private final FutureShufflesCache shufflesCache;
    private final IEventCollector eventCollector;

    public Bundle(ScanContext scanContext, FutureShufflesCache shufflesCache, IEventCollector eventCollector) {
      this.scanContext = checkNotNull(scanContext);
      this.shufflesCache = checkNotNull(shufflesCache);
      this.eventCollector = checkNotNull(eventCollector);
    }

    public ScanContext getScanContext() {
      return scanContext;
    }

    public FutureShufflesCache getShufflesCache() {
      return shufflesCache;
    }

    IEventCollector getEventCollector() {
      return eventCollector;
    }
  }

  enum State {
    IDLE,
    SCANNING
  }

}
