package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanRecord;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.os.Build;

import com.kontakt.sdk.android.ble.discovery.ScanErrors;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.IOException;
import java.util.Collection;

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
abstract class MonitorCallbackL extends ScanCallback implements BleScanCallback {

  static MonitorCallbackL wrap(BleScanCallback scanCallback) {
    return new MonitorCallbackL((MonitorCallback) scanCallback) {
    };
  }

  private final MonitorCallback wrappedScanCallback;

  private MonitorCallbackL(MonitorCallback scanCallback) {
    SDKPreconditions.checkNotNull(scanCallback, "Wrapped scan callback is null");
    this.wrappedScanCallback = scanCallback;
  }

  @Override
  public void onScanResult(int callbackType, ScanResult result) {
    if (callbackType != ScanSettings.CALLBACK_TYPE_ALL_MATCHES) {
      return;
    }

    final ScanRecord scanRecord = result.getScanRecord();
    if (scanRecord == null) {
      return;
    }
    onLeScan(result.getDevice(), result.getRssi(), scanRecord.getBytes());
  }

  @Override
  public void onLeScan(BluetoothDevice device, int rssi, byte[] scanRecord) {
    wrappedScanCallback.onLeScan(device, rssi, scanRecord);
  }

  @Override
  public void onScanFailed(int errorCode) {
    final String message = ScanErrors.getMessage(errorCode);
    Logger.e(message);
    wrappedScanCallback.onScanError(new ScanError(message));
  }

  @Override
  public Collection<InternalProximityListener> getMonitoringListeners() {
    return wrappedScanCallback.getMonitoringListeners();
  }

  @Override
  public void addListener(InternalProximityListener proximityListener) {
    wrappedScanCallback.addListener(proximityListener);
  }

  @Override
  public void removeListener(InternalProximityListener proximityListener) {
    wrappedScanCallback.removeListener(proximityListener);
  }

  @Override
  public void close() throws IOException {
    wrappedScanCallback.close();
  }

  void onMonitorStarted() {
    wrappedScanCallback.onMonitorCycleStart();
  }

  void onMonitorStopped() {
    wrappedScanCallback.onMonitorCycleStop();
  }

}
