package com.kontakt.sdk.android.ble.discovery.secure_profile;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.discovery.FramePayload;
import com.kontakt.sdk.android.ble.discovery.ShuffledSecureProfileResolver;
import com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver;
import com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver;
import com.kontakt.sdk.android.common.model.Model;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import java.util.Locale;

import static com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver.SECURE_PROFILE_INSTANCE_ID_START_INDEX;
import static com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver.SECURE_PROFILE_NAMESPACE_ID_START_INDEX;

public class ShuffledDevicePayloadResolver extends PayloadResolver implements ShuffledSecureProfileResolver.ResolveCallback {

  private static final byte PAYLOAD_IDENTIFIER = 0x01;

  private final ShuffledSecureProfileResolver shuffleResolver;

  private static final NamespaceIdResolver NAMESPACE_RESOLVER = new NamespaceIdResolver(SECURE_PROFILE_NAMESPACE_ID_START_INDEX);
  private static final InstanceIdResolver INSTANCE_ID_RESOLVER = new InstanceIdResolver(SECURE_PROFILE_INSTANCE_ID_START_INDEX);

  public ShuffledDevicePayloadResolver(final FutureShufflesCache shufflesCache) {
    super(PAYLOAD_IDENTIFIER);
    this.shuffleResolver = new ShuffledSecureProfileResolver(this, shufflesCache);
  }

  // used for tests
  ShuffledDevicePayloadResolver(final ShuffledSecureProfileResolver shuffleResolver) {
    super(PAYLOAD_IDENTIFIER);
    this.shuffleResolver = shuffleResolver;
  }

  @Override
  public void resolve(final FramePayload payload) {
    if (payload == null) {
      return;
    }

    // Get MAC address, name & service data
    final String macAddress = payload.getAddress();
    final String name = getName(payload);
    final byte[] serviceData = getServiceData(payload);

    // Parse properties
    final Model model = Model.fromCode(serviceData[3]);
    final String firmware = parseFirmwareVersion(serviceData);
    final int batteryLevel = serviceData[6];
    final int txPower = serviceData[7];
    final String namespace = parseNamespace(serviceData);
    final String shuffledInstanceId = parseShuffledInstanceId(serviceData);

    // Build shuffled secure profile object
    final SecureProfile secureProfile = new SecureProfile.Builder()
        .macAddress(macAddress)
        .name(name)
        .model(model)
        .firmwareRevision(firmware)
        .batteryLevel(batteryLevel)
        .txPower(txPower)
        .namespace(namespace)
        .instanceId(shuffledInstanceId)
        .shuffled(true)
        .build();

    // Try to resolve and wait for callback
    shuffleResolver.resolve(secureProfile);
  }

  private String parseFirmwareVersion(final byte[] serviceData) {
    final int firmwareVersionMajor = serviceData[4];
    final int firmwareVersionMinor = serviceData[5];

    if (firmwareVersionMajor < 0 || firmwareVersionMinor < 0) {
      return "";
    }
    return String.format(Locale.getDefault(), "%d.%d", firmwareVersionMajor, firmwareVersionMinor);
  }

  private String parseNamespace(final byte[] serviceData) {
    return NAMESPACE_RESOLVER.parse(serviceData);
  }

  private String parseShuffledInstanceId(final byte[] serviceData) {
    return INSTANCE_ID_RESOLVER.parse(serviceData);
  }

  void handleLostEvent(ISecureProfile profile) {
    shuffleResolver.onDeviceLost(profile);
  }

  void disableShuffleResolver() {
    shuffleResolver.disable();
  }

  @Override
  public void onResolved(ISecureProfile profile) {
    // Shuffled device resolved - notify
    onPayloadResolved(new SecureProfile.Builder(profile).build());
  }

}
