package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.bluetooth.BluetoothDevice;
import android.util.Log;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.discovery.AbstractDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.discovery.Validator;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import java.util.List;

public class IBeaconDiscoverer extends AbstractDeviceDiscoverer<IBeaconRegion, IBeaconDevice, IBeaconFilter> {

  private static final String TAG = IBeaconDiscoverer.class.getSimpleName();
  private final Validator<IBeaconDevice, IBeaconRegion> validator = new RegionValidator();
  private final IBeaconParser parser;

  public IBeaconDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract, FutureShufflesCache cache) {
    super(discoveryContract, scanContext, scanContext.getIBeaconRegions(), scanContext.getIBeaconFilters(), cache);
    this.parser = new IBeaconParser(scanContext);
  }

  @Override
  protected BluetoothDeviceEvent createEvent(EventType eventType, IBeaconRegion iBeaconRegion, List<IBeaconDevice> deviceList) {
    return new IBeaconDeviceEvent(eventType, iBeaconRegion, deviceList);
  }

  @Override
  protected void onBeforeDeviceLost(IBeaconDevice device) {
    parser.clearRssiCalculation(device.getAddress().hashCode());
  }

  @Override
  public void performDiscovery(BluetoothDevice device, int rssi, byte[] scanRecord) {
    if (!parser.isEnabled() || !parser.isValidIBeaconFrame(scanRecord)) {
      return;
    }

    Log.d(TAG, String.format("Parsing iBeacon: %s", device.getAddress()));
    parser.parseScanRecord(scanRecord);

    if (parser.getFrameData().size() == 0 || !parser.isManufacturerDataValid()) {
      return;
    }

    IBeaconDevice iBeaconDevice = parser.getIBeaconDevice(device, rssi);

    Log.d(TAG, String.format("Parsed iBeacon: %s", device.getAddress()));
    notifyDevicePresent(device.getAddress().hashCode(), System.currentTimeMillis());

    if (iBeaconDevice.isShuffled()) {
      resolveShuffled(iBeaconDevice);
    } else {
      onShuffleResolved(iBeaconDevice);
    }
  }

  @Override
  protected void onShuffleResolved(RemoteBluetoothDevice device) {
    if (!(device instanceof IBeaconDevice)) return;

    IBeaconDevice iBeaconDevice = (IBeaconDevice) device;

    final IBeaconRegion iBeaconRegion = extractRegion(iBeaconDevice);
    if (iBeaconRegion == null) {
      return;
    }

    if (device.getProximity() != Proximity.UNKNOWN) {
      notifySpacePresent(iBeaconRegion.hashCode(), System.currentTimeMillis());

      ReplacingArrayList<IBeaconDevice> iBeaconList = getDevicesInSpace(iBeaconRegion);
      if (iBeaconList == null) {
        iBeaconList = new ReplacingArrayList<>();
        insertDevicesIntoSpace(iBeaconRegion, iBeaconList);
        onSpaceEnteredEvent(iBeaconRegion);
      }

      if (!applyFilters(iBeaconDevice)) {
        return;
      }

      if (iBeaconList.addOrReplace(iBeaconDevice)) {
        onDeviceDiscoveredEvent(iBeaconRegion, iBeaconDevice);
      } else {
        onDevicesUpdatedEvent(iBeaconRegion, iBeaconList);
      }
    }
  }

  private IBeaconRegion extractRegion(final IBeaconDevice iBeaconDevice) {
    for (final IBeaconRegion iBeaconRegion : getSpaceSet()) {
      if (validator.isValid(iBeaconDevice, iBeaconRegion)) {
        return iBeaconRegion;
      }
    }
    return null;
  }

  @Override
  public void disable() {
    parser.disable();
    super.disable();
  }
}
