package com.kontakt.sdk.android.ble.discovery;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconParser;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.LimitedLinkedHashMap;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public abstract class Parser<Device> {

  protected static final int CACHE_SIZE = 50;
  private static final byte[] ACCELEROMETER_MANUFACTURER_DATA_PREFIX = new byte[] { (byte) 0xFD };
  private static final byte[] EDDYSTONE_PACKET_SERVICE_DATA_TYPE_PREFIX = new byte[] { (byte) 0xAA, (byte) 0xFE };

  protected final Map<Integer, Device> devicesCache = new LimitedLinkedHashMap<>(CACHE_SIZE);
  protected final Map<FrameDataType, byte[]> frameData = new HashMap<>();
  protected final HashCodeBuilder hashCodeBuilder = HashCodeBuilder.init();
  protected boolean isEnabled = true;
  protected final RssiCalculator rssiCalculator;

  protected Parser(ScanContext scanContext) {
    this.rssiCalculator = scanContext.getRssiCalculator();
  }

  protected abstract void disable();

  public Map<FrameDataType, byte[]> getFrameData() {
    return frameData;
  }

  public boolean isEnabled() {
    return isEnabled;
  }

  public void clearRssiCalculation(int key) {
    if (rssiCalculator != null) {
      rssiCalculator.clear(key);
    }
  }

  protected void extractFrameData(final byte[] scanRecord, final Map<FrameDataType, byte[]> frameData) {
    int index = 0;
    while (index < scanRecord.length) {
      final int length = scanRecord[index++];

      if (length == 0) {
        break;
      }

      final int type = ConversionUtils.asInt(scanRecord[index]);

      if (type == 0) {
        break;
      }

      final byte[] data = Arrays.copyOfRange(scanRecord, index + 1, index + length);

      FrameDataType frameDataType = FrameDataType.fromByte(type);
      if (frameDataType.isServiceData()) {
        extractServiceData(frameData, data);
      } else if (frameDataType.isManufacturerSpecificData()) {
        extractManufacturerSpecificData(frameData, data);
      } else if (frameDataType != FrameDataType.UNKNOWN) {
        frameData.put(frameDataType, data);
      }

      index += length;
    }
  }

  private void extractServiceData(Map<FrameDataType, byte[]> frameData, byte[] data) {
    if (ConversionUtils.doesArrayBeginWith(data, ScanResponse.SCAN_RESPONSE_D00D_PREFIX) || ConversionUtils.doesArrayBeginWith(data,
        ScanResponse.SCAN_RESPONSE_PDI_PREFIX)) {
      frameData.put(FrameDataType.SCAN_RESPONSE_SERVICE_DATA, data);
    } else if (ConversionUtils.doesArrayBeginWith(data, EDDYSTONE_PACKET_SERVICE_DATA_TYPE_PREFIX)) {
      frameData.put(FrameDataType.EDDYSTONE_PACKET_SERVICE_DATA, data);
    }
  }

  private void extractManufacturerSpecificData(Map<FrameDataType, byte[]> frameData, byte[] data) {
    if (ConversionUtils.doesArrayBeginWith(data, IBeaconParser.MANUFACTURER_DATA_IBEACON_PREFIX)) {
      frameData.put(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA, data);
    } else if (ConversionUtils.doesArrayBeginWith(data, ACCELEROMETER_MANUFACTURER_DATA_PREFIX)) {
      frameData.put(FrameDataType.ACCELEROMETER_BEACON_MANUFACTURER_SPECIFIC_DATA, data);
    }
  }
}
