package com.kontakt.sdk.android.ble.connection;

import android.content.Context;

import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.KontaktCloudFactory;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudError;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

@SuppressWarnings("WeakerAccess")
class SyncableKontaktDeviceConnectionImpl implements SyncableKontaktDeviceConnection {

  /**
   * <p>
   * Abstract class used as listener to notify about updating device and syncing with REST API
   * <p>
   * <p>
   * Current implementation does nothing
   */
  public abstract static class SyncWriteListener {

    /**
     * The constant SYNC_NOT_IMPLEMENTED_YET.
     */
    public static final int SYNC_NOT_IMPLEMENTED_YET = -1;

    /**
     * Invoked when beacon characteristics manipulation failed.
     * <p>
     * If so no attempt will be done to update REST API
     *
     * @deprecated use {@link #onWriteFailed(ErrorCause)} instead
     */
    @Deprecated
    public void onWriteFailed() {
    }

    /**
     * Invoked when beacon characteristics manipulation failed.
     * <p>
     * If so no attempt will be done to update REST API
     *
     * @param cause {@link ErrorCause}
     */
    public void onWriteFailed(ErrorCause cause) {
    }

    /**
     * <p>
     * Invoked when beacon non secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e the exception object
     */
    public void onSyncFailed(KontaktCloudException e) {
    }

    /**
     * <p>
     * Invoked when beacon non secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e the error object
     */
    public void onSyncFailed(CloudError e) {
    }

    /**
     * <p>
     * Invoked when beacon beacon secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e        the exception object
     * @param response encrypted response from beacon
     */
    public void onSyncFailed(KontaktCloudException e, String response) {

    }

    /**
     * <p>
     * Invoked when beacon beacon secure characteristics manipulation was success, but invoking REST API failed
     * <p>
     * <p>
     * No attempts to restore beacon old characteristics will be done
     *
     * @param e              the error object
     * @param secureResponse encrypted response from beacon
     */
    public void onSyncFailed(CloudError e, Config secureResponse) {

    }

    /**
     * <p>
     * Invoked when beacon characteristics manipulation was success and invoking REST API succeed
     */
    public void onSuccess() {
    }
  }

  private KontaktDeviceConnectionImpl kontaktDeviceConnection;
  private KontaktCloud kontaktCloud;

  /**
   * Instantiates a new Syncable Kontakt device connection.
   *
   * @param context               the context
   * @param remoteBluetoothDevice the beacon device
   * @param connectionListener    the connection listener
   */
  SyncableKontaktDeviceConnectionImpl(Context context, RemoteBluetoothDevice remoteBluetoothDevice,
                                      KontaktDeviceConnectionImpl.ConnectionListener connectionListener) {
    kontaktDeviceConnection = new KontaktDeviceConnectionImpl(context, remoteBluetoothDevice, connectionListener);
    kontaktCloud = KontaktCloudFactory.create();
  }

  @Override
  public synchronized boolean connectToDevice() {
    return kontaktDeviceConnection.connect();
  }

  @Override
  public synchronized boolean isConnectedToDevice() {
    return kontaktDeviceConnection.isConnected();
  }

  @Override
  public synchronized void close() {
    kontaktDeviceConnection.close();
  }

  @Override
  public synchronized RemoteBluetoothDevice getDevice() {
    return kontaktDeviceConnection.getDevice();
  }

  @Override
  public synchronized boolean isDeviceAuthenticated() {
    return kontaktDeviceConnection.isAuthenticated();
  }

  @Override
  public void applySecureConfig(Config secureConfig, SyncWriteListener syncWriteListener) {
    kontaktDeviceConnection.applySecureConfig(secureConfig.getSecureRequest(),
        new InternalSecureBeaconWriteListener(syncWriteListener, kontaktCloud, getDeviceUniqueId()));
  }

  private String getDeviceUniqueId() {
    return getDevice().getUniqueId();
  }
}
