package com.kontakt.sdk.android.ble.configuration;

import java.util.concurrent.TimeUnit;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;

/**
 * Describes time of single period describing how long Android device should scan Bluetooth devices
 * and how long it should stay in an idle state. The period consists of two sub-periods:
 * activePeriod - the period describing how long Android device performs scanning in search of
 * Bluetooth devices
 * and
 * passivePeriod - the period describing how long the Android device should be
 * in idle state in terms of bluetooth devices scan.
 *
 * The scan period parameters specify scan behaviour in terms of switching between 2 states: the
 * scanning state and the idle state. If the scan period has passive sub-period set to 0 then
 * the scanning state works instantly without a break.
 *
 * If the passive scan sub-period has positive value then scan behaves in the monitoring manner switching
 * between scanning and idle state.
 *
 * There is a validation for the scan period when including to
 * {@link ScanContext}.
 * The active scan period must be a positive value equal or longer than 3000 ms whereas
 * the passive scan period must be a positive value equal or higher than 2000ms or equal to 0.
 */
public class ScanPeriod {

  public static final long MINIMAL_ACTIVE_SCAN_PERIOD = TimeUnit.SECONDS.toMillis(3);
  public static final long MINIMAL_PASSIVE_SCAN_PERIOD = TimeUnit.SECONDS.toMillis(2);

  private final long activePeriod;
  private final long passivePeriod;

  /**
   * The constant RANGING. Active scan will run constantly.
   */
  public static final ScanPeriod RANGING = new ScanPeriod(TimeUnit.SECONDS.toMillis(60), 0L);

  /**
   * The constant MONITORING. Equals 8 seconds of active scanning and 30 seconds of waiting.
   */
  public static final ScanPeriod MONITORING = new ScanPeriod(TimeUnit.SECONDS.toMillis(8), TimeUnit.SECONDS.toMillis(30));

  /**
   * Instantiates a new Scan period.
   *
   * @param activePeriodMillis the active period [ms]
   * @param passivePeriodMillis the passive period [ms]
   * @return instance of ScanPeriod
   */
  public static ScanPeriod create(final long activePeriodMillis, final long passivePeriodMillis) {
    checkArgument(activePeriodMillis >= MINIMAL_ACTIVE_SCAN_PERIOD, "Active Scan Period must be greater than " + MINIMAL_ACTIVE_SCAN_PERIOD);
    checkArgument(passivePeriodMillis >= MINIMAL_PASSIVE_SCAN_PERIOD, "Passive Scan Period must be greater than " + MINIMAL_PASSIVE_SCAN_PERIOD);
    return new ScanPeriod(activePeriodMillis, passivePeriodMillis);
  }

  private ScanPeriod(final long activePeriod, final long passivePeriod) {
    this.activePeriod = activePeriod;
    this.passivePeriod = passivePeriod;
  }

  /**
   * Provides active scan period [ms].
   *
   * @return the active period
   */
  public long getActivePeriod() {
    return activePeriod;
  }

  /**
   * Provides passive period [ms].
   *
   * @return the passive period
   */
  public long getPassivePeriod() {
    return passivePeriod;
  }

  /**
   * Provides sum of both active and passive period [ms].
   *
   * @return the full period - sum of the active and passive sub-periods.
   */
  public long getFullPeriod() {
    return activePeriod + passivePeriod;
  }
}
