package com.kontakt.sdk.android.ble.configuration;

import android.os.Build;

import com.kontakt.sdk.android.ble.configuration.changable.BLEScannerTypeProvider;
import com.kontakt.sdk.android.ble.configuration.changable.ConstantIntensitivityTypeProvider;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.discovery.secure_profile.PayloadResolver;
import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.ble.manager.configuration.FiltersConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.GeneralConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.SpacesConfigurator;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.rssi.RssiCalculators;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validate;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateCacheFileName;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateDeviceUpdateCallbackInterval;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateEddystoneFiltersCount;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateIBeaconFiltersCount;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateIBeaconRegionsCount;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateMonitoringSyncInterval;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateNamespacesCount;
import static com.kontakt.sdk.android.ble.configuration.ScanContextValidator.validateResolveShuffleInterval;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableSet;

/**
 * Provides scan-specific parameters influencing scan performance profiles of BLE devices to be found.
 */
@SuppressWarnings("WeakerAccess")
public class ScanContext {

  /**
   * Default settings which works well on most modern devices
   */
  public static final ScanContext DEFAULT = new ScanContext.Builder().build();

  static final long DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL = 3000;
  static final String DEFAULT_CACHE_FILE_NAME = "resolved.che";
  static final int DEFAULT_RESOLVE_INTERVAL = 3;
  static final int DEFAULT_MONITORING_SYNC_INTERVAL = 10;
  static final boolean DEFAULT_MONITORING_ENABLED = true;

  private final ScanPeriod scanPeriod;
  private final ScanMode scanMode;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final ForceScanConfiguration forceScanConfiguration;
  private final RssiCalculator rssiCalculator;
  private final Set<IBeaconRegion> iBeaconRegions;
  private final Set<IEddystoneNamespace> eddystoneNamespaces;
  private final Set<EddystoneFrameType> eddystoneTriggerFrameTypes;
  private final List<IBeaconFilter> iBeaconFilters;
  private final List<EddystoneFilter> eddystoneFilters;
  private final Set<DeviceProfile> observedProfiles;
  private final List<PayloadResolver> customSecureProfilePayloadResolvers;
  private final String cacheFileName;
  private final int resolveInterval;
  private final int monitoringSyncInterval;
  private final boolean monitoringEnabled;
  private final long deviceUpdateCallbackInterval;

  private BLEScannerTypeProvider bleScannerTypeProvider;

  ScanContext(Builder builder) {
    this.forceScanConfiguration = builder.forceScanConfiguration;
    this.scanPeriod = builder.scanPeriod;
    this.scanMode = builder.scanMode;
    this.activityCheckConfiguration = builder.activityCheckConfiguration;
    this.deviceUpdateCallbackInterval = builder.deviceUpdateCallbackInterval;
    this.rssiCalculator = builder.rssiCalculator;
    this.observedProfiles = builder.observedProfiles;
    this.eddystoneTriggerFrameTypes = builder.eddystoneTriggerFrameTypes;
    this.customSecureProfilePayloadResolvers = builder.customSecureProfilePayloadResolvers;
    this.iBeaconRegions = builder.iBeaconRegions;
    this.eddystoneNamespaces = builder.eddystoneNamespaces;
    this.iBeaconFilters = builder.iBeaconFilters;
    this.eddystoneFilters = builder.eddystoneFilters;
    this.cacheFileName = builder.cacheFileName;
    this.resolveInterval = builder.resolveInterval;
    this.monitoringSyncInterval = builder.monitoringSyncInterval;
    this.monitoringEnabled = builder.monitoringEnabled;
    this.bleScannerTypeProvider = builder.bleScannerTypeProvider;
  }

  public ActivityCheckConfiguration getActivityCheckConfiguration() {
    return activityCheckConfiguration;
  }

  public Set<DeviceProfile> getObservedProfiles() {
    return observedProfiles;
  }

  public ForceScanConfiguration getForceScanConfiguration() {
    return forceScanConfiguration;
  }

  public ScanPeriod getScanPeriod() {
    return scanPeriod;
  }

  public ScanMode getScanMode() {
    return scanMode;
  }

  public long getDeviceUpdateCallbackInterval() {
    return deviceUpdateCallbackInterval;
  }

  public RssiCalculator getRssiCalculator() {
    return rssiCalculator;
  }

  public Set<EddystoneFrameType> getEddystoneFrameTypes() {
    return unmodifiableSet(eddystoneTriggerFrameTypes);
  }

  public Set<IBeaconRegion> getIBeaconRegions() {
    return unmodifiableSet(iBeaconRegions);
  }

  public Set<IEddystoneNamespace> getEddystoneNamespaces() {
    return unmodifiableSet(eddystoneNamespaces);
  }

  public List<IBeaconFilter> getIBeaconFilters() {
    return unmodifiableList(iBeaconFilters);
  }

  public List<EddystoneFilter> getEddystoneFilters() {
    return unmodifiableList(eddystoneFilters);
  }

  public List<PayloadResolver> getCustomSecureProfilePayloadResolvers() {
    return unmodifiableList(customSecureProfilePayloadResolvers);
  }

  public String getCacheFileName() {
    return cacheFileName;
  }

  public int getResolveInterval() {
    return resolveInterval;
  }

  public int getMonitoringSyncInterval() {
    return monitoringSyncInterval;
  }

  public boolean isMonitoringEnabled() {
    return monitoringEnabled;
  }

  public BLEScannerTypeProvider getBLEScannerTypeProvider() { return bleScannerTypeProvider; }

  public static final class Builder implements GeneralConfigurator, SpacesConfigurator, FiltersConfigurator {

    ScanMode scanMode = ScanMode.BALANCED;
    ScanPeriod scanPeriod = ScanPeriod.RANGING;
    ForceScanConfiguration forceScanConfiguration = ForceScanConfiguration.DISABLED;
    ActivityCheckConfiguration activityCheckConfiguration = ActivityCheckConfiguration.DEFAULT;
    RssiCalculator rssiCalculator = RssiCalculators.DEFAULT;
    Set<DeviceProfile> observedProfiles = EnumSet.allOf(DeviceProfile.class);
    Set<EddystoneFrameType> eddystoneTriggerFrameTypes = EnumSet.noneOf(EddystoneFrameType.class);
    Set<IBeaconRegion> iBeaconRegions = new HashSet<>();
    Set<IEddystoneNamespace> eddystoneNamespaces = new HashSet<>();
    List<IBeaconFilter> iBeaconFilters = new ArrayList<>();
    List<EddystoneFilter> eddystoneFilters = new ArrayList<>();
    List<PayloadResolver> customSecureProfilePayloadResolvers = new ArrayList<>();
    int resolveInterval = DEFAULT_RESOLVE_INTERVAL;
    int monitoringSyncInterval = DEFAULT_MONITORING_SYNC_INTERVAL;
    long deviceUpdateCallbackInterval = DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL;
    boolean monitoringEnabled = DEFAULT_MONITORING_ENABLED;
    String cacheFileName = DEFAULT_CACHE_FILE_NAME;

    BLEScannerTypeProvider bleScannerTypeProvider = new ConstantIntensitivityTypeProvider(false);

    public Builder() {
    }

    public Builder(ScanContext context) {
      this.scanMode = context.getScanMode();
      this.scanPeriod = context.getScanPeriod();
      this.forceScanConfiguration = context.getForceScanConfiguration();
      this.activityCheckConfiguration = context.getActivityCheckConfiguration();
      this.deviceUpdateCallbackInterval = context.getDeviceUpdateCallbackInterval();
      this.rssiCalculator = context.getRssiCalculator();
      this.observedProfiles = context.getObservedProfiles();
      this.eddystoneTriggerFrameTypes = context.getEddystoneFrameTypes();
      this.customSecureProfilePayloadResolvers.clear();
      this.customSecureProfilePayloadResolvers.addAll(context.getCustomSecureProfilePayloadResolvers());
      this.cacheFileName = context.getCacheFileName();
      this.resolveInterval = context.getResolveInterval();
      this.monitoringSyncInterval = context.getMonitoringSyncInterval();
      this.monitoringEnabled = context.isMonitoringEnabled();
      this.iBeaconRegions.clear();
      this.iBeaconRegions.addAll(context.getIBeaconRegions());
      this.iBeaconFilters.clear();
      this.iBeaconFilters.addAll(context.getIBeaconFilters());
      this.eddystoneNamespaces.clear();
      this.eddystoneNamespaces.addAll(context.getEddystoneNamespaces());
      this.eddystoneFilters.clear();
      this.eddystoneFilters.addAll(context.getEddystoneFilters());
      this.bleScannerTypeProvider = context.bleScannerTypeProvider;
    }

    @Override
    public GeneralConfigurator scanPeriod(ScanPeriod scanPeriod) {
      checkNotNull(scanPeriod, "Monitor period cannot be null");
      this.scanPeriod = scanPeriod;
      return this;
    }

    @Override
    public GeneralConfigurator scanMode(ScanMode scanMode) {
      checkNotNull(scanMode, "Scan mode is null");
      this.scanMode = scanMode;
      return this;
    }

    @Override
    public GeneralConfigurator forceScanConfiguration(ForceScanConfiguration forceScanConfiguration) {
      checkNotNull(forceScanConfiguration, "By default ForceScanConfiguration is disabled");
      //Android N prevents apps from from starting and stopping mor than 5 times in 30 seconds.
      //https://github.com/AltBeacon/android-beacon-library/issues/418
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
        this.forceScanConfiguration = ForceScanConfiguration.DISABLED;
      } else {
        this.forceScanConfiguration = forceScanConfiguration;
      }
      return this;
    }

    @Override
    public GeneralConfigurator activityCheckConfiguration(ActivityCheckConfiguration activityCheckConfiguration) {
      checkNotNull(activityCheckConfiguration, "Beacon activity check is null.");
      this.activityCheckConfiguration = activityCheckConfiguration;
      return this;
    }

    @Override
    public GeneralConfigurator cacheFileName(String name) {
      checkNotNull(name, "Cache file name cannot be null!");
      this.cacheFileName = name;
      return this;
    }

    @Override
    public GeneralConfigurator monitoringEnabled(boolean enabled) {
      this.monitoringEnabled = enabled;
      return this;
    }

    @Override
    public GeneralConfigurator monitoringSyncInterval(int intervalInSeconds) {
      this.monitoringSyncInterval = intervalInSeconds;
      return this;
    }

    @Override
    public GeneralConfigurator resolveShuffledInterval(int intervalInSeconds) {
      this.resolveInterval = intervalInSeconds;
      return this;
    }

    @Override
    public GeneralConfigurator deviceUpdateCallbackInterval(long intervalInMillis) {
      this.deviceUpdateCallbackInterval = intervalInMillis;
      return this;
    }

    @Override
    public GeneralConfigurator rssiCalculator(RssiCalculator rssiCalculator) {
      checkNotNull(rssiCalculator, "RssiCalculator can't be null");
      this.rssiCalculator = rssiCalculator;
      return this;
    }

    @Override
    public GeneralConfigurator eddystoneFrameTypes(Collection<EddystoneFrameType> eddystoneFrameTypes) {
      checkNotNull(eddystoneFrameTypes, "Eddystone trigger frames");
      this.eddystoneTriggerFrameTypes.clear();
      this.eddystoneTriggerFrameTypes.addAll(eddystoneFrameTypes);
      return this;
    }

    @Override
    public GeneralConfigurator secureProfilePayloadResolver(PayloadResolver payloadResolver) {
      checkNotNull(payloadResolver);
      secureProfilePayloadResolvers(singletonList(payloadResolver));
      return this;
    }

    @Override
    public GeneralConfigurator secureProfilePayloadResolvers(Collection<PayloadResolver> payloadResolvers) {
      checkNotNull(payloadResolvers, "Resolvers collection is null");
      this.customSecureProfilePayloadResolvers.clear();
      this.customSecureProfilePayloadResolvers.addAll(payloadResolvers);
      return this;
    }

    @Override
    public SpacesConfigurator iBeaconRegion(IBeaconRegion region) {
      checkNotNull(region);
      iBeaconRegions(singletonList(region));
      return this;
    }

    @Override
    public SpacesConfigurator iBeaconRegions(Collection<IBeaconRegion> regions) {
      checkNotNull(regions, "Regions collection is null");
      this.iBeaconRegions.clear();
      this.iBeaconRegions.addAll(regions);
      return this;
    }

    @Override
    public SpacesConfigurator eddystoneNamespace(IEddystoneNamespace namespace) {
      checkNotNull(namespace);
      eddystoneNamespaces(singletonList(namespace));
      return this;
    }

    @Override
    public SpacesConfigurator eddystoneNamespaces(Collection<IEddystoneNamespace> namespaces) {
      checkNotNull(namespaces, "Eddystone namespaces are null.");
      this.eddystoneNamespaces.clear();
      this.eddystoneNamespaces.addAll(namespaces);
      return this;
    }

    @Override
    public Set<IBeaconRegion> getIBeaconRegions() {
      return unmodifiableSet(iBeaconRegions);
    }

    @Override
    public Set<IEddystoneNamespace> getEddystoneNamespaces() {
      return unmodifiableSet(eddystoneNamespaces);
    }

    @Override
    public FiltersConfigurator iBeaconFilter(IBeaconFilter filter) {
      checkNotNull(filter);
      iBeaconFilters(singletonList(filter));
      return this;
    }

    @Override
    public FiltersConfigurator iBeaconFilters(Collection<IBeaconFilter> filters) {
      checkNotNull(filters, "Filters are null.");
      this.iBeaconFilters.clear();
      this.iBeaconFilters.addAll(filters);
      return this;
    }

    @Override
    public FiltersConfigurator eddystoneFilter(EddystoneFilter filter) {
      checkNotNull(filter);
      eddystoneFilters(singletonList(filter));
      return this;
    }

    @Override
    public FiltersConfigurator eddystoneFilters(Collection<EddystoneFilter> filters) {
      checkNotNull(filters, "Filters are null.");
      this.eddystoneFilters.clear();
      this.eddystoneFilters.addAll(filters);
      return this;
    }

    @Override
    public List<IBeaconFilter> getIBeaconFilters() {
      return unmodifiableList(iBeaconFilters);
    }

    @Override
    public List<EddystoneFilter> getEddystoneFilters() {
      return unmodifiableList(eddystoneFilters);
    }

    @Override
    public void clearAll() {
      this.iBeaconFilters(Collections.<IBeaconFilter>emptyList());
      this.eddystoneFilters(Collections.<EddystoneFilter>emptyList());
    }

    public Builder observedProfiles(Set<DeviceProfile> observedProfiles) {
      checkNotNull(observedProfiles, "Observed profiles can't be null");
      this.observedProfiles = observedProfiles;
      return this;
    }

    public Builder addObservedProfile(DeviceProfile profile) {
      checkNotNull(profile, "Observed profiles can't be null");
      this.observedProfiles.add(profile);
      return this;
    }

    public Builder removeObservedProfile(DeviceProfile profile) {
      checkNotNull(profile, "Observed profiles can't be null");
      this.observedProfiles.remove(profile);
      return this;
    }

    public Builder bLEScannerTypeProvider(BLEScannerTypeProvider bleScannerTypeProvider){
      checkNotNull(bleScannerTypeProvider, "BLEScannerTypeProvider cannot be null");
      this.bleScannerTypeProvider = bleScannerTypeProvider;
      return this;
    }

    public ScanContext build() {
      validate(scanPeriod);
      validateMonitoringSyncInterval(monitoringSyncInterval);
      validateResolveShuffleInterval(resolveInterval);
      validateDeviceUpdateCallbackInterval(deviceUpdateCallbackInterval);
      validateCacheFileName(cacheFileName);
      validateEddystoneFiltersCount(eddystoneFilters);
      validateIBeaconFiltersCount(iBeaconFilters);
      validateIBeaconRegionsCount(iBeaconRegions);
      validateNamespacesCount(eddystoneNamespaces);
      validate(activityCheckConfiguration);
      validate(forceScanConfiguration);
      validate(activityCheckConfiguration, scanPeriod);

      if (iBeaconRegions.isEmpty()) {
        iBeaconRegions.add(BeaconRegion.EVERYWHERE);
      }
      if (eddystoneNamespaces.isEmpty()) {
        eddystoneNamespaces.add(EddystoneNamespace.EVERYWHERE);
      }

      return new ScanContext(this);
    }
  }
}
