package com.kontakt.sdk.android.common.util;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

/**
 * Various array utilities.
 */
public class ArrayUtils {

  private static final int[] EMPTY_INT_ARRAY = new int[0];
  private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];

  /**
   * Convert 4B array to integer
   *
   * @param array source
   * @return integer result.
   */
  public static int byteArrayToInt(byte[] array) {
    ByteBuffer buffer = ByteBuffer.wrap(array);
    buffer.order(ByteOrder.LITTLE_ENDIAN);
    return buffer.getInt();
  }

  /**
   * Concats two byte arrays
   * @param first First array
   * @param second Second array
   * @return concated array
   */
  public static byte[] concat(byte[] first, byte[] second) {
    int aLength = first.length;
    int bLength = second.length;
    byte[] c = new byte[aLength + bLength];
    System.arraycopy(first, 0, c, 0, aLength);
    System.arraycopy(second, 0, c, aLength, bLength);
    return c;
  }

  /**
   * Concats two String arrays
   * @param first First array
   * @param second Second array
   * @return concated array
   */
  public static String[] concat(String[] first, String[] second) {
    int aLength = first.length;
    int bLength = second.length;
    String[] c = new String[aLength + bLength];
    System.arraycopy(first, 0, c, 0, aLength);
    System.arraycopy(second, 0, c, aLength, bLength);
    return c;
  }

  /**
   * <p>Converts an array of object Bytes to primitives.
   *
   * <p>This method returns {@code null} for a {@code null} input array.
   *
   * @param array a {@code Byte} array, may be {@code null}
   * @return a {@code byte} array, {@code null} if null array input
   * @throws NullPointerException if array content is {@code null}
   */
  public static byte[] toPrimitive(Byte[] array) {
    if (array == null) {
      return null;
    } else if (array.length == 0) {
      return EMPTY_BYTE_ARRAY;
    }
    final byte[] result = new byte[array.length];
    for (int i = 0; i < array.length; i++) {
      result[i] = array[i];
    }
    return result;
  }

  /**
   * <p>Converts an array of object Integers to primitives.
   *
   * <p>This method returns {@code null} for a {@code null} input array.
   *
   * @param array a {@code Integer} array, may be {@code null}
   * @return a {@code int} array, {@code null} if null array input
   * @throws NullPointerException if array content is {@code null}
   */
  public static int[] toPrimitive(final Integer[] array) {
    if (array == null) {
      return null;
    } else if (array.length == 0) {
      return EMPTY_INT_ARRAY;
    }
    final int[] result = new int[array.length];
    for (int i = 0; i < array.length; i++) {
      result[i] = array[i];
    }
    return result;
  }

  public static void printByteArray(byte[] data, String s) {
    System.out.println(s);
    for (byte datum : data) {
      System.out.print(String.format("%02X ", datum));
    }
    System.out.println();
  }

  public static String binaryStringOfByte(byte b){
    return Integer.toBinaryString((b & 0xFF) + 0x100).substring(1);
  }

}
