package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;

import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * An abstract representation of configuration which {@link Device} must be updated with.
 * It's simply a container of device's settings that can be configured.
 * <br><br>
 * We can obtain configs via {@link KontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link Config.Builder}.
 */
public class Config implements Parcelable {

  public static final Creator<Config> CREATOR = new Creator<Config>() {
    @Override
    public Config createFromParcel(Parcel source) {
      return new Config(source);
    }

    @Override
    public Config[] newArray(int size) {
      return new Config[size];
    }
  };

  private static final int TX_POWER_MIN_VALUE = 0;
  private static final int TX_POWER_MAX_VALUE = 7;

  private static final int INTERVAL_MIN_VALUE = 20;
  private static final int INTERVAL_MAX_VALUE = 10240;

  public static final int TEMPERATURE_OFFSET_DISABLED_VALUE = Byte.MAX_VALUE;
  private static final int TEMPERATURE_OFFSET_MIN_VALUE = Byte.MIN_VALUE;
  private static final int TEMPERATURE_OFFSET_MAX_VALUE = Byte.MAX_VALUE - 1;

  private String uniqueId;
  private UUID proximity;
  private int major = -1;
  private int minor = -1;
  private int txPower = -1;
  private int interval = -1;
  private String namespace;
  private String url;
  private String instanceId;
  private List<DeviceProfile> profiles;
  private List<PacketType> packets;
  private Boolean shuffled;
  private String name;
  private String password;
  private PowerSaving powerSaving;
  private List<Integer> rssi1m;
  private List<Integer> rssi0m;

  // Kontakt Telemetry
  private int temperatureOffset = TEMPERATURE_OFFSET_DISABLED_VALUE;

  // Gateway
  private Network gatewayNetwork;

  // Custom fields
  private Map<String, String> customConfiguration;

  // Secure configuration
  /**
   * Base64 encoded request data that need to be sent to the device.
   */
  @SerializedName("config") private String secureRequest;

  /**
   * Base64 encoded response data received from the device after configuration.
   */
  private String secureResponse;

  /**
   * Timestamp of when the response was received.
   */
  private long secureResponseTime;

  public static Builder builder() {
    return new Builder();
  }

  Config(Builder builder) {
    uniqueId = builder.uniqueId;
    interval = builder.interval;
    txPower = builder.txPower;
    instanceId = builder.instanceId;
    url = builder.url;
    namespace = builder.namespace;
    name = builder.name;
    proximity = builder.proximity;
    major = builder.major;
    minor = builder.minor;
    profiles = builder.profiles;
    packets = builder.packets;
    shuffled = builder.shuffled;
    password = builder.password;
    secureRequest = builder.secureRequest;
    secureResponse = builder.secureResponse;
    secureResponseTime = builder.secureResponseTime;
    powerSaving = builder.powerSaving;
    rssi1m = builder.rssi1m;
    rssi0m = builder.rssi0m;
    temperatureOffset = builder.temperatureOffset;
    gatewayNetwork = builder.gatewayNetwork;
    customConfiguration = builder.customConfiguration;
  }

  protected Config(Parcel in) {
    this.uniqueId = in.readString();
    this.proximity = (UUID) in.readSerializable();
    this.major = in.readInt();
    this.minor = in.readInt();
    this.txPower = in.readInt();
    this.interval = in.readInt();
    this.namespace = in.readString();
    this.url = in.readString();
    this.instanceId = in.readString();
    this.profiles = new ArrayList<>();
    in.readList(this.profiles, DeviceProfile.class.getClassLoader());
    this.packets = new ArrayList<>();
    in.readList(this.packets, PacketType.class.getClassLoader());
    this.shuffled = (Boolean) in.readValue(Boolean.class.getClassLoader());
    this.name = in.readString();
    this.password = in.readString();
    this.powerSaving = in.readParcelable(PowerSaving.class.getClassLoader());
    this.rssi1m = new ArrayList<>();
    in.readList(this.rssi1m, Integer.class.getClassLoader());
    this.rssi0m = new ArrayList<>();
    in.readList(this.rssi0m, Integer.class.getClassLoader());
    this.secureRequest = in.readString();
    this.secureResponse = in.readString();
    this.secureResponseTime = in.readLong();
    this.temperatureOffset = in.readInt();
    this.gatewayNetwork = in.readParcelable(Network.class.getClassLoader());

    int customConfigurationSize = in.readInt();
    if (customConfigurationSize != 0) {
      this.customConfiguration = new HashMap<>(customConfigurationSize);
      for (int i = 0; i < customConfigurationSize; i++) {
        String key = in.readString();
        String value = in.readString();
        this.customConfiguration.put(key, value);
      }
    }
  }

  private Config() {
    this(new Builder());
  }

  public void applyConfig(final Config config) {
    SDKPreconditions.checkNotNull(config, "config data cannot be null");
    if (config.interval != -1) {
      interval = config.interval;
    }
    if (config.txPower != -1) {
      txPower = config.txPower;
    }
    if (config.instanceId != null) {
      instanceId = config.instanceId;
    }
    if (config.url != null) {
      url = config.url;
    }
    if (config.namespace != null) {
      namespace = config.namespace;
    }
    if (config.name != null) {
      name = config.name;
    }
    if (config.proximity != null) {
      proximity = config.proximity;
    }
    if (config.major != -1) {
      major = config.major;
    }
    if (config.minor != -1) {
      minor = config.minor;
    }
    if (config.profiles != null) {
      profiles = config.profiles;
    }
    if (config.packets != null) {
      packets = config.packets;
    }
    if (config.shuffled != null) {
      shuffled = config.shuffled;
    }
    if (config.password != null) {
      password = config.password;
    }
    if (config.powerSaving != null) {
      powerSaving = config.powerSaving;
    }
    if (config.rssi1m != null) {
      rssi1m = config.rssi1m;
    }
    if (config.rssi0m != null) {
      rssi0m = config.rssi0m;
    }
    if (config.gatewayNetwork != null) {
      gatewayNetwork = config.gatewayNetwork;
    }
    if (config.customConfiguration != null) {
      customConfiguration = config.customConfiguration;
    }
  }

  public void applySecureConfig(final Config secureConfig) {
    SDKPreconditions.checkNotNull(secureConfig, "secure config data cannot be null");
    if (secureConfig.secureRequest != null) {
      secureRequest = secureConfig.secureRequest;
    }
    if (secureConfig.secureResponse != null) {
      secureResponse = secureConfig.secureResponse;
    }
    if (secureConfig.secureResponseTime > 0) {
      secureResponseTime = secureConfig.secureResponseTime;
    }
  }

  /**
   * Specifies request data that could be sent to the device.
   *
   * @param secureRequest secure request data.
   */
  public void applySecureRequest(final String secureRequest) {
    SDKPreconditions.checkNotNullOrEmpty(secureRequest, "secure request cannot be either null or empty");
    this.secureRequest = secureRequest;
  }

  /**
   * Specifies response data received from the device after successful configuration update.
   *
   * @param secureResponse     secure response data received from the device.
   * @param secureResponseTime timestamp of when the response was received.
   */
  public void applySecureResponse(final String secureResponse, final long secureResponseTime) {
    SDKPreconditions.checkNotNull(secureResponse, "secure response cannot be null");
    SDKPreconditions.checkState(secureResponseTime > 0, "secure response timestamp cannot be negative");
    this.secureResponse = secureResponse;
    this.secureResponseTime = secureResponseTime;
  }

  public void changePassword(final String password) {
    this.password = password;
  }

  /**
   * Checks if the config is secure or not. By secure config we can understood config with firmware
   * version higher or equal to 4.0 and initialized secure request value.
   *
   * @return {@code true} if the config is secure, {@code false} otherwise.
   */
  public boolean isSecureConfig() {
    return secureRequest != null && !"".equals(secureRequest);
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public UUID getProximity() {
    return proximity;
  }

  public int getMajor() {
    return major;
  }

  public int getMinor() {
    return minor;
  }

  public int getTxPower() {
    return txPower;
  }

  public int getInterval() {
    return interval;
  }

  public String getNamespace() {
    return namespace;
  }

  /**
   * Returns the URL value encoded in hexed string.
   *
   * @return the encoded URL value.
   */
  public String getHexUrl() {
    if (url != null) {
      return EddystoneUtils.toHexString(EddystoneUtils.serializeUrl(url));
    }
    return null;
  }

  public String getUrl() {
    return url;
  }

  public String getInstanceId() {
    return instanceId;
  }

  public List<DeviceProfile> getProfiles() {
    if (profiles != null) {
      return Collections.unmodifiableList(profiles);
    }
    return null;
  }

  public List<PacketType> getPackets() {
    if (packets != null) {
      return Collections.unmodifiableList(packets);
    }
    return null;
  }

  public Boolean getShuffled() {
    return shuffled;
  }

  public boolean isShuffled() {
    if (shuffled == null) {
      return false;
    }
    return shuffled;
  }

  public PowerSaving getPowerSaving() {
    return powerSaving;
  }

  public String getName() {
    return name;
  }

  public String getPassword() {
    return password;
  }

  public List<Integer> getRssi1m() {
    return rssi1m;
  }

  public List<Integer> getRssi0m() {
    return rssi0m;
  }

  public int getTemperatureOffset() {
    return temperatureOffset;
  }

  public Network getGatewayNetwork() {
    return gatewayNetwork;
  }

  public Map<String, String> getCustomConfiguration() {
    return customConfiguration;
  }

  /**
   * Returns Base64 encoded request data that need to be sent to the device.
   *
   * @return the secure request data.
   */
  public String getSecureRequest() {
    return secureRequest;
  }

  /**
   * Returns Base64 encoded response data received from the device after successful configuration update.
   *
   * @return the secure response data.
   */
  public String getSecureResponse() {
    return secureResponse;
  }

  /**
   * Returns a timestamp of when the secure response was received.
   *
   * @return the timestamp value.
   */
  public long getSecureResponseTime() {
    return secureResponseTime;
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof Config)) {
      return false;
    }
    final Config config = (Config) object;

    return SDKEqualsBuilder.start()
        .equals(uniqueId, config.uniqueId)
        .equals(proximity, config.proximity)
        .equals(major, config.major)
        .equals(minor, config.minor)
        .equals(txPower, config.txPower)
        .equals(interval, config.interval)
        .equals(namespace, config.namespace)
        .equals(url, config.url)
        .equals(instanceId, config.instanceId)
        .equals(name, config.name)
        .equals(password, config.password)
        .equals(profiles, config.profiles)
        .equals(packets, config.packets)
        .equals(powerSaving, config.powerSaving)
        .equals(temperatureOffset, config.temperatureOffset)
        .equals(gatewayNetwork, config.gatewayNetwork)
        .equals(customConfiguration, config.customConfiguration)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(uniqueId)
        .append(proximity)
        .append(major)
        .append(minor)
        .append(txPower)
        .append(interval)
        .append(namespace)
        .append(url)
        .append(instanceId)
        .append(shuffled)
        .append(name)
        .append(password)
        .append(profiles)
        .append(packets)
        .append(powerSaving)
        .append(temperatureOffset)
        .append(gatewayNetwork)
        .append(customConfiguration)
        .build();
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.uniqueId);
    dest.writeSerializable(this.proximity);
    dest.writeInt(this.major);
    dest.writeInt(this.minor);
    dest.writeInt(this.txPower);
    dest.writeInt(this.interval);
    dest.writeString(this.namespace);
    dest.writeString(this.url);
    dest.writeString(this.instanceId);
    dest.writeList(this.profiles);
    dest.writeList(this.packets);
    dest.writeValue(this.shuffled);
    dest.writeString(this.name);
    dest.writeString(this.password);
    dest.writeParcelable(this.powerSaving, flags);
    dest.writeList(this.rssi1m);
    dest.writeList(this.rssi0m);
    dest.writeString(this.secureRequest);
    dest.writeString(this.secureResponse);
    dest.writeLong(this.secureResponseTime);
    dest.writeInt(this.temperatureOffset);
    dest.writeParcelable(this.gatewayNetwork, flags);

    int customConfigurationSize = this.customConfiguration != null ? this.customConfiguration.size() : 0;
    dest.writeInt(customConfigurationSize);
    if (customConfigurationSize != 0) {
      for (Map.Entry<String, String> entry : this.customConfiguration.entrySet()) {
        dest.writeString(entry.getKey());
        dest.writeString(entry.getValue());
      }
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public String toString() {
    return "Config{" +
        "uniqueId='" + uniqueId + '\'' +
        ", proximity=" + proximity +
        ", major=" + major +
        ", minor=" + minor +
        ", txPower=" + txPower +
        ", interval=" + interval +
        ", namespace='" + namespace + '\'' +
        ", url='" + url + '\'' +
        ", instanceId='" + instanceId + '\'' +
        ", profiles=" + profiles +
        ", packets=" + packets +
        ", shuffled=" + shuffled +
        ", name='" + name + '\'' +
        ", password='" + password + '\'' +
        ", powerSaving=" + powerSaving +
        ", rssi1m=" + rssi1m +
        ", rssi0m=" + rssi0m +
        ", secureRequest='" + secureRequest + '\'' +
        ", secureResponse='" + secureResponse + '\'' +
        ", secureResponseTime=" + secureResponseTime +
        ", temperatureOffset=" + temperatureOffset +
        ", customConfiguration=" + customConfiguration +
        ", network=" + gatewayNetwork +
        '}';
  }

  /**
   * Builder class that is used to build {@link Config} instances from values configured by the setters.
   */
  public static class Builder {
    String uniqueId;
    int interval = -1;
    int txPower = -1;
    String instanceId;
    String url;
    String namespace;
    String name;
    UUID proximity;
    int major = -1;
    int minor = -1;
    List<DeviceProfile> profiles = new ArrayList<>();
    List<PacketType> packets = new ArrayList<>();
    Boolean shuffled;
    String password;
    String secureRequest;
    String secureResponse;
    long secureResponseTime;
    PowerSaving powerSaving;
    List<Integer> rssi1m;
    List<Integer> rssi0m;
    int temperatureOffset = TEMPERATURE_OFFSET_DISABLED_VALUE;
    Network gatewayNetwork;
    Map<String, String> customConfiguration;

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder interval(final int interval) {
      if (interval >= INTERVAL_MIN_VALUE && interval <= INTERVAL_MAX_VALUE) {
        this.interval = interval;
      }
      return this;
    }

    public Builder txPower(final int txPower) {
      if (txPower >= TX_POWER_MIN_VALUE && txPower <= TX_POWER_MAX_VALUE) {
        this.txPower = txPower;
      }
      return this;
    }

    public Builder instanceId(final String instanceId) {
      this.instanceId = instanceId;
      return this;
    }

    public Builder url(final String url) {
      if (url != null) {
        final boolean isHexUrl = EddystoneUtils.isStringOnlyHex(url);
        if (isHexUrl) {
          this.url = EddystoneUtils.fromHexedUrlToUrl(url);
        } else {
          this.url = url;
        }
      } else {
        this.url = null;
      }
      return this;
    }

    public Builder namespace(final String namespace) {
      this.namespace = namespace;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder proximity(final UUID proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder major(final int major) {
      if (major >= 0) {
        this.major = major;
      }
      return this;
    }

    public Builder minor(final int minor) {
      if (minor >= 0) {
        this.minor = minor;
      }
      return this;
    }

    public Builder profiles(final Collection<DeviceProfile> profiles) {
      if (profiles == null) {
        return this;
      }

      for (DeviceProfile profile : profiles) {
        SDKPreconditions.checkNotNull(profile, "profiles cannot contain null value");
      }
      this.profiles.clear();
      for (DeviceProfile profile : profiles) {
        if (!this.profiles.contains(profile)) {
          this.profiles.add(profile);
        }
      }
      return this;
    }

    public Builder packets(final Collection<PacketType> packets) {
      if (packets == null) {
        return this;
      }

      for (PacketType packet : packets) {
        SDKPreconditions.checkNotNull(packet, "packets cannot contain null value");
      }
      this.packets.clear();
      for (PacketType packet : packets) {
        if (!this.packets.contains(packet)) {
          this.packets.add(packet);
        }
      }
      return this;
    }

    public Builder shuffled(final boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    public Builder password(final String password) {
      this.password = password;
      return this;
    }

    public Builder secureRequest(final String secureRequest) {
      this.secureRequest = secureRequest;
      return this;
    }

    public Builder secureResponse(final String secureResponse) {
      this.secureResponse = secureResponse;
      return this;
    }

    public Builder secureResponseTime(final long secureResponseTime) {
      if (secureResponseTime >= 0) {
        this.secureResponseTime = secureResponseTime;
      }
      return this;
    }

    public Builder powerSaving(PowerSaving powerSaving) {
      this.powerSaving = powerSaving;
      return this;
    }

    public Builder rssi1m(List<Integer> rssi1m) {
      this.rssi1m = rssi1m;
      return this;
    }

    public Builder rssi0m(List<Integer> rssi0m) {
      this.rssi0m = rssi0m;
      return this;
    }

    public Builder temperatureOffset(final int temperatureOffset) {
      if (temperatureOffset >= TEMPERATURE_OFFSET_MIN_VALUE && temperatureOffset <= TEMPERATURE_OFFSET_MAX_VALUE) {
        this.temperatureOffset = temperatureOffset;
      }
      return this;
    }

    public Builder gatewayNetwork(final Network network) {
      this.gatewayNetwork = network;
      return this;
    }

    public Builder customConfiguration(final Map<String, String> customConfiguration) {
      if (customConfiguration == null) {
        return this;
      }
      this.customConfiguration = customConfiguration;
      return this;
    }

    public Builder addCustomField(final String key, final String value) {
      checkNotNull(key, "Key cannot be null");
      checkNotNull(value, "Value cannot be null");

      if (this.customConfiguration == null) {
        this.customConfiguration = new HashMap<>();
      }
      this.customConfiguration.put(key, value);
      return this;
    }

    public Config build() {
      // resolve interleaving packets before building config
      if (packets.size() == 0) {
        if (profiles.contains(DeviceProfile.IBEACON)) {
          packets.add(PacketType.IBEACON);
        }
        if (profiles.contains(DeviceProfile.EDDYSTONE)) {
          packets.add(PacketType.EDDYSTONE_UID);
          packets.add(PacketType.EDDYSTONE_URL);
          packets.add(PacketType.EDDYSTONE_TLM);
        }
      }
      return new Config(this);
    }

  }

}
