package com.kontakt.sdk.android.cloud.api.executor.receivers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ReceiversApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ReceiversService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Receiver;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

/**
 * Request executor provided by {@link ReceiversApi}. Use this class if you want to update
 * receivers through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   // define updated receiver object
 *   kontaktCloud.receivers().update(ID)
 *      .with(updatedReceiver)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that receiver must be specified so invocation of {@code with} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class UpdateReceiverRequestExecutor extends RequestExecutor<String> {

  private final ReceiversService receiversService;
  private final UUID id;
  private Receiver receiver;

  /**
   * Constructs request executor initialized with corresponding service class and receiver ID.
   *
   * @param receiversService the receivers API service.
   * @param id the receiver's identifier.
   */
  public UpdateReceiverRequestExecutor(ReceiversService receiversService, UUID id) {
    this.receiversService = receiversService;
    this.id = id;
  }

  /**
   * Specifies receiver data to update. The method invocation is obligatory while using this request executor.
   *
   * @param receiver updated receiver data.
   * @return this request executor.
   */
  public UpdateReceiverRequestExecutor with(final Receiver receiver) {
    checkNotNull(receiver, "receiver cannot be null");
    this.receiver = receiver;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return receiversService.updateReceiver(params());
  }

  private void checkPreconditions() {
    checkState(receiver != null, "cannot update receiver - specify receiver data to update");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Receivers.RECEIVER_ID_PARAMETER, id.toString());
    if (receiver.getDeviceUniqueId() != null) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, receiver.getDeviceUniqueId());
    }
    if (receiver.getPlaceId() != null) {
      params.put(CloudConstants.Receivers.PLACE_ID_PARAMETER, receiver.getPlaceId().toString());
    }
    if (receiver.getCoordinates() != null) {
      params.put(CloudConstants.Receivers.COORDINATES_PARAMETER, receiver.getCoordinates().toString());
    }
    return params;
  }
}
