package com.kontakt.sdk.android.ble.manager.internal;

import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.CloudError;
import com.kontakt.sdk.android.cloud.response.CloudHeaders;
import com.kontakt.sdk.android.cloud.response.paginated.Namespaces;
import com.kontakt.sdk.android.cloud.response.paginated.Proximities;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Namespace;
import com.kontakt.sdk.android.common.model.ProximityId;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Class is responsible for resolving shuffled iBeacon regions or Eddystone namespaces using Kontakt Cloud.
 */
@SuppressWarnings("WeakerAccess")
abstract class SpacesResolver<Space> {

  interface ResolvingStatusListener {
    void onSuccess();

    void onError(String message);
  }

  protected final KontaktCloud cloud;
  protected final Map<String, Space> toResolve = new HashMap<>();
  protected final Collection<Space> resolved = new ArrayList<>();
  protected final Map<String, Space> cache = new HashMap<>();
  protected ResolvingStatusListener resolvingStatusListener;
  protected int maxResults = 50;
  private boolean isFinishedSuccessfully;

  SpacesResolver(KontaktCloud cloud) {
    this.cloud = checkNotNull(cloud);
  }

  public void resolve(ScanContext scanContext, ResolvingStatusListener onSpacesResolvedListener) {
    this.resolvingStatusListener = checkNotNull(onSpacesResolvedListener);
    isFinishedSuccessfully = false;
    clear();
    divideSpaces(getSpacesFromScanContext(scanContext));
    if (toResolve.isEmpty()) {
      invokeSuccessCallback();
      Logger.d("No shuffled spaces need resolving.");
      return;
    }
    maxResults = toResolve.size() + 50;
    fetchSpacesFromCloud();
  }

  public void clear() {
    toResolve.clear();
    resolved.clear();
  }

  void clearCache() {
    cache.clear();
  }

  void resetFinishedStatus() {
    isFinishedSuccessfully = false;
  }

  boolean isFinishedSuccessfully() {
    return isFinishedSuccessfully;
  }

  Collection<Space> getResolvedSpaces() {
    return resolved;
  }

  protected abstract Collection<Space> getSpacesFromScanContext(ScanContext scanContext);

  protected abstract void divideSpaces(Collection<Space> spaces);

  protected abstract void fetchSpacesFromCloud();

  void invokeSuccessCallback() {
    isFinishedSuccessfully = true;
    resolvingStatusListener.onSuccess();
  }

  void invokeErrorCallback(String message) {
    resolvingStatusListener.onError(message);
    Logger.e("Error while fetching spaces from the cloud: " + message);
  }

  static final class RegionsResolver extends SpacesResolver<IBeaconRegion> {

    RegionsResolver(KontaktCloud cloud) {
      super(cloud);
    }

    @Override
    protected Collection<IBeaconRegion> getSpacesFromScanContext(ScanContext scanContext) {
      return scanContext.getIBeaconRegions();
    }

    @Override
    protected void divideSpaces(Collection<IBeaconRegion> iBeaconRegions) {
      for (IBeaconRegion region : iBeaconRegions) {
        UUID proximity = region.getProximity();
        UUID secureProximity = region.getSecureProximity();

        if (secureProximity != null) {
          if (cache.containsKey(secureProximity.toString())) {
            resolved.add(cache.get(secureProximity.toString()));
            continue;
          }

          if (proximity != null && cache.containsKey(proximity.toString())) {
            //We already have resolved proximity
            resolved.add(cache.get(proximity.toString()));
            continue;
          }

          toResolve.put(secureProximity.toString(), region);
        } else {
          resolved.add(region);
        }
      }
    }

    @Override
    protected void fetchSpacesFromCloud() {
      Logger.d("Fetching regions from the cloud...");
      cloud.proximities().fetch().maxResult(maxResults).execute(new CloudCallback<Proximities>() {
        @Override
        public void onSuccess(final Proximities proximities, CloudHeaders headers) {
          resolveRegions(proximities.getContent());
          invokeSuccessCallback();
          Logger.d("Successfully fetched regions from Kontakt Cloud");
        }

        @Override
        public void onError(CloudError error) {
          invokeErrorCallback(error.getMessage());
        }
      });
    }

    void resolveRegions(List<ProximityId> proximityIds) {
      for (ProximityId fetchedRegion : proximityIds) {
        IBeaconRegion region = toResolve.get(fetchedRegion.getProximityUUID().toString());
        if (region != null
            && fetchedRegion.isShuffled()
            && fetchedRegion.getProximityUUID().equals(region.getSecureProximity())) {
          IBeaconRegion shuffledRegion = createNewRegion(region, fetchedRegion.getProximityUUID(), fetchedRegion.getSecureProximityUUID());
          resolved.add(shuffledRegion);
          cache.put(region.getSecureProximity().toString(), shuffledRegion);
        }
      }
    }

    private IBeaconRegion createNewRegion(IBeaconRegion oldRegion, UUID normalUUID, UUID secureUUID) {
      return new BeaconRegion.Builder()
          .identifier(oldRegion.getIdentifier())
          .proximity(normalUUID)
          .secureProximity(secureUUID)
          .major(oldRegion.getMajor())
          .minor(oldRegion.getMinor())
          .build();
    }
  }

  static final class NamespacesResolver extends SpacesResolver<IEddystoneNamespace> {

    NamespacesResolver(KontaktCloud cloud) {
      super(cloud);
    }

    @Override
    protected Collection<IEddystoneNamespace> getSpacesFromScanContext(ScanContext scanContext) {
      return scanContext.getEddystoneNamespaces();
    }

    @Override
    protected void divideSpaces(Collection<IEddystoneNamespace> namespaces) {
      for (IEddystoneNamespace namespace : namespaces) {
        String normalNamespace = namespace.getNamespace();
        String secureNamespace = namespace.getSecureNamespace();

        if (secureNamespace != null) {
          if (cache.containsKey(secureNamespace)) {
            resolved.add(cache.get(secureNamespace));
            continue;
          }

          if (normalNamespace != null && cache.containsKey(normalNamespace)) {
            //We already have resolved namespace
            resolved.add(cache.get(normalNamespace));
            continue;
          }

          toResolve.put(secureNamespace, namespace);
        } else {
          resolved.add(namespace);
        }
      }
    }

    @Override
    protected void fetchSpacesFromCloud() {
      Logger.d("Fetching namespaces from the cloud...");
      cloud.namespaces().fetch().maxResult(maxResults).execute(new CloudCallback<Namespaces>() {
        @Override
        public void onSuccess(Namespaces namespaces, CloudHeaders headers) {
          resolveNamespaces(namespaces.getContent());
          invokeSuccessCallback();
          Logger.d("Successfully fetched namespaces from Kontakt Cloud");
        }

        @Override
        public void onError(CloudError error) {
          invokeErrorCallback(error.getMessage());
        }
      });
    }

    void resolveNamespaces(List<Namespace> namespaces) {
      for (Namespace fetchedNamespace : namespaces) {
        IEddystoneNamespace namespace = toResolve.get(fetchedNamespace.getNamespaceId());
        if (namespace != null
            && fetchedNamespace.isShuffled()
            && fetchedNamespace.getNamespaceId().equals(namespace.getSecureNamespace())) {
          IEddystoneNamespace shuffledNamespace =
              createNewNamespace(namespace, fetchedNamespace.getNamespaceId(), fetchedNamespace.getSecureNamespaceId());
          resolved.add(shuffledNamespace);
          cache.put(namespace.getSecureNamespace(), shuffledNamespace);
        }
      }
    }

    private IEddystoneNamespace createNewNamespace(IEddystoneNamespace oldSpace, String normalNamespace, String secureNamespace) {
      return new EddystoneNamespace.Builder()
          .identifier(oldSpace.getIdentifier())
          .namespace(normalNamespace)
          .secureNamespace(secureNamespace)
          .instanceId(oldSpace.getInstanceId())
          .build();
    }
  }
}
