package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.os.Build;
import android.util.Log;
import android.util.SparseLongArray;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.util.SafeSparseLongArray;
import com.kontakt.sdk.android.common.profile.ISecureProfile;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewDiscovered;
import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewLost;
import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewUpdated;
import static java.lang.System.currentTimeMillis;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class SecureProfileDiscoverer implements BluetoothDeviceDiscoverer, ParseListener {

  private static final String TAG = SecureProfileDiscoverer.class.getSimpleName();

  private final Map<String, ISecureProfile> cache; //Mac Address -> ISecureProfile
  private final SecureProfileParser parser;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final SparseLongArray secureProfileTimestampArray;
  private final DiscoveryContract discoveryContract;
  private final long deviceUpdateIntervalCallback;
  private long lastUpdateCallbackMillis;

  public SecureProfileDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract, FutureShufflesCache shufflesCache) {
    this.deviceUpdateIntervalCallback = scanContext.getDeviceUpdateCallbackInterval();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.secureProfileTimestampArray = new SafeSparseLongArray();
    this.discoveryContract = discoveryContract;
    this.cache = new ConcurrentHashMap<>();

    // Initialize parser with listener
    this.parser = new SecureProfileParser(scanContext, shufflesCache);
    this.parser.addListener(this);
  }

  // used for tests
  SecureProfileDiscoverer(
      ScanContext scanContext,
      DiscoveryContract discoveryContract,
      SparseLongArray secureProfileTimestampArray,
      Map<String, ISecureProfile> cache,
      FutureShufflesCache shufflesCache) {
    this.discoveryContract = discoveryContract;
    this.deviceUpdateIntervalCallback = scanContext.getDeviceUpdateCallbackInterval();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.secureProfileTimestampArray = secureProfileTimestampArray;
    this.cache = cache;

    // Initialize parser with listener
    this.parser = new SecureProfileParser(scanContext, shufflesCache);
    this.parser.addListener(this);
  }

  @Override
  public void performDiscovery(BluetoothDevice device, int rssi, byte[] scanResult) {
    // Validate frame
    if (!parser.isValidSecureProfileFrame(scanResult)) {
      return;
    }

    // Parse and wait for callback
    Log.d(TAG, String.format("Parsing secure profile with mac: %s", device.getAddress()));

    parser.parseSecureProfile(device, rssi, scanResult);
  }

  @Override
  public void onParsed(ISecureProfile profile) {
    notifySecureProfilePresent(profile);
    handleResolved(profile);
  }

  private void handleResolved(ISecureProfile secureProfile) {
    boolean isDiscovered = cache.containsKey(secureProfile.getMacAddress());
    Log.d(TAG, String.format("Parsed secure profile: %s", secureProfile.toString()));
    cache.put(secureProfile.getMacAddress(), secureProfile);

    if (!isDiscovered) {
      onDiscovered(secureProfile);
    } else {
      onUpdated();
    }
  }

  @Override
  public void evictInactiveDevices(long currentTimeMillis) {
    final long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();

    for (ISecureProfile secureProfile : new ArrayList<>(cache.values())) {
      long timestamp = getSecureProfileTimestamp(secureProfile);
      if (timestamp != -1L) {
        long threshold = currentTimeMillis - timestamp;
        if (threshold > inactivityTimeout) {
          cache.remove(secureProfile.getMacAddress());
          clearBeforeLost(secureProfile);
          onLost(secureProfile);
          removeSecureProfileTimestamp(secureProfile);
        }
      }
    }
  }

  private void notifySecureProfilePresent(ISecureProfile secureProfile) {
    secureProfileTimestampArray.put(secureProfile.getMacAddress().hashCode(), currentTimeMillis());
  }

  private void onDiscovered(ISecureProfile secureProfile) {
    discoveryContract.onEvent(createNewDiscovered(secureProfile, currentTimeMillis()));
  }

  private void onUpdated() {
    long currentTimeMillis = currentTimeMillis();
    Log.d(TAG, "On updated ");
    if (currentTimeMillis - lastUpdateCallbackMillis > deviceUpdateIntervalCallback) {
      discoveryContract.onEvent(createNewUpdated(new ArrayList<>(cache.values()), currentTimeMillis));
      lastUpdateCallbackMillis = currentTimeMillis;
    }
    else{
      Log.d(
          TAG, String.format("Skipping Secure update event, deviceUpdateInterval: %d currentTimeMillis: %d lastUpdateCallback: %d curr - last = %d",
          deviceUpdateIntervalCallback, currentTimeMillis, lastUpdateCallbackMillis, currentTimeMillis - lastUpdateCallbackMillis)
      );
    }
  }

  private void onLost(ISecureProfile secureProfile) {
    discoveryContract.onEvent(createNewLost(secureProfile, currentTimeMillis()));
    parser.handleLostEvent(secureProfile);
  }

  private void clearBeforeLost(ISecureProfile secureProfile) {
    parser.clearRssiCalculation(secureProfile.getMacAddress().hashCode());
  }

  private long getSecureProfileTimestamp(ISecureProfile secureProfile) {
    return secureProfileTimestampArray.get(secureProfile.getMacAddress().hashCode(), -1L);
  }

  private void removeSecureProfileTimestamp(ISecureProfile secureProfile) {
    int index = secureProfileTimestampArray.indexOfKey(secureProfile.getMacAddress().hashCode());
    if (index >= 0) {
      secureProfileTimestampArray.removeAt(index);
    }
  }

  @Override
  public void disable() {
    parser.disable();
  }

}
