package com.kontakt.sdk.android.ble.discovery.secure_profile;

import com.kontakt.sdk.android.ble.discovery.PropertyResolver;
import com.kontakt.sdk.android.ble.spec.Acceleration;
import com.kontakt.sdk.android.ble.spec.DeviceDataLoggerStatus;
import com.kontakt.sdk.android.ble.spec.KontaktTelemetry;
import com.kontakt.sdk.android.ble.spec.TelemetryError;

import static com.kontakt.sdk.android.common.util.ConversionUtils.asFloatFromLittleEndianBytes;
import static com.kontakt.sdk.android.common.util.ConversionUtils.asInt;
import static com.kontakt.sdk.android.common.util.ConversionUtils.asIntFromLittleEndianBytes;
import static com.kontakt.sdk.android.common.util.ConversionUtils.extractSubdata;

final class KontaktTLMResolver implements PropertyResolver<KontaktTelemetry> {

  // FIELD IDENTIFIERS

  private static final byte BASIC_SYSTEM_HEALTH_IDENTIFIER = 0x01;
  private static final byte ACCELEROMETER_IDENTIFIER = 0x02;
  private static final byte SCANNING_IDENTIFIER = 0x03;
  private static final byte MORE_SYSTEM_HEALTH_IDENTIFIER = 0x04;
  private static final byte SENSORS_IDENTIFIER = 0x05;
  private static final byte RAW_ACCELEROMETER_IDENTIFIER = 0x06;
  private static final byte MOVEMENT_THRESHOLD_EVENT_IDENTIFIER = 0x07;
  private static final byte DOUBLE_TAP_IDENTIFIER = 0x08;
  private static final byte TAP_EVENT_IDENTIFIER = 0x09;
  private static final byte LIGHT_LEVEL_IDENTIFIER = 0x0A;
  private static final byte TEMPERATURE_IDENTIFIER = 0x0B;
  private static final byte BATTERY_IDENTIFIER = 0x0C;
  private static final byte BUTTON_CLICK_IDENTIFIER = 0x0D;
  private static final byte BUTTON_DOUBLE_CLICK_IDENTIFIER = 0x0E;
  private static final byte UTC_TIME_IDENTIFIER = 0x0F;
  private static final byte LOGGING_ENABLED_IDENTIFIER = 0x10;
  private static final byte IDENTIFIED_BUTTON_CLICK_IDENTIFIER = 0x11;
  private static final byte HUMIDITY_IDENTIFIER = 0x12;
  private static final byte TEMPERATURE_16_BIT_IDENTIFIER = 0x13;
  private static final byte BLE_CHANNEL_IDENTIFIER = 0x14;
  private static final byte GPIO_PIN_STATE = 0x15;
  private static final byte MOVEMENT_EVENT_IDENTIFIER = 0x16;
  private static final byte AIR_PRESSURE_IDENTIFIER = 0x17;
  private static final byte PIR_IDENTIFIER =  0x18;
  private static final byte TWO_BUTTON_INFO_IDENTIFIER = 0x19;

  private static final byte VENDOR_SPECIFIC_PAYLOAD_IDENTIFIER = (byte) 0XFF;
  private static final byte AIR_QUALITY_IDENTIFIER = 0x1a;
  private static final byte ROOM_NUMBER_IDENTIFIER = 0x1b;
  private static final byte OCCUPANCY_IDENTIFIER = 0x1c;

  // FIELD LENGTHS

  private static final int BASIC_SYSTEM_HEALTH_FIELD_LENGTH = 6;
  private static final int ACCELEROMETER_FIELD_LENGTH = 9;
  private static final int SCANNING_FIELD_LENGTH = 5;
  private static final int MORE_SYSTEM_HEALTH_FIELD_LENGTH = 10;
  private static final int SENSORS_FIELD_LENGTH = 3;
  private static final int RAW_ACCELEROMETER_LENGTH = 5;
  private static final int MOVEMENT_THRESHOLD_EVENT_LENGTH  = 3;
  private static final int DOUBLE_TAP_EVENT_LENGTH = 3;
  private static final int TAP_EVENT_LENGTH = 3;
  private static final int LIGHT_LEVEL_FIELD_LENGTH = 2;
  private static final int TEMPERATURE_FIELD_LENGTH = 2;
  private static final int BATTERY_FIELD_LENGTH = 2;
  private static final int BUTTON_CLICK_FIELD_LENGTH = 3;
  private static final int BUTTON_DOUBLE_CLICK_FIELD_LENGTH  = 3;
  private static final int UTC_TIME_FIELD_LENGTH = 5;
  private static final int LOGGING_ENABLED_FIELD_LENGTH = 2;
  private static final int IDENTIFIED_BUTTON_CLICK_FIELD_LENGTH = 4;
  private static final int HUMIDITY_FIELD_LENGTH = 2;
  private static final int TEMPERATURE_16_BIT_FIELD_LENGTH = 3;
  private static final int BLE_CHANNEL_FIELD_LENGTH = 2;
  private static final int GPIO_PIN_STATE_FIELD_LENGTH = 3;
  private static final int MOVEMENT_EVENT_FIELD_LENGTH = 4;
  private static final int AIR_PRESSURE_FIELD_LENGTH = 5;
  private static final int PIR_DETECTION_FIELD_LENGTH = 3;
  private static final int TWO_BUTTON_INFO_FIELD_LENGTH = 5;

  private static final int AIR_QUALITY_FIELD_LENGTH = 2;
  private static final int ROOM_NUMBER_FIELD_LENGTH = 3;
  private static final int OCCUPANCY_FIELD_LENGTH = 2;

  private static final int FIELDS_PAYLOAD_OFFSET = 3;
  private static final int VENDOR_SPECIFIC_PAYLOAD_OFFSET = 3;

  @Override
  public KontaktTelemetry parse(byte[] packet) {
    if (packet == null || packet.length == 0) {
      return null;
    }
    final KontaktTelemetry.Builder builder = new KontaktTelemetry.Builder();
    byte[] fieldsData = extractSubdata(packet, FIELDS_PAYLOAD_OFFSET, packet.length - FIELDS_PAYLOAD_OFFSET);
    iterateOverFields(builder, fieldsData);
    return builder.build();
  }

  private void iterateOverFields(KontaktTelemetry.Builder builder, byte[] fieldsData){
    while (fieldsData != null && fieldsData.length > 0) {
      final int fieldLength = asInt(fieldsData[0]);
      final byte[] field = extractSubdata(fieldsData, 1, fieldLength);
      if (field == null) break;
      final byte identifier = field[0];
      resolve(identifier, builder, field);
      fieldsData = extractSubdata(fieldsData, field.length + 1, fieldsData.length - (field.length + 1));
    }
  }

  private void resolve(byte identifier, KontaktTelemetry.Builder builder, byte[] field){
    resolveBasicSystemHealth(identifier, builder, field);
    resolveAccelerometerData(identifier, builder, field);
    resolveScanningData(identifier, builder, field);
    resolveMoreSystemHealth(identifier, builder, field);
    resolveSensors(identifier, builder, field);
    resolveRawAccelerometer(identifier, builder, field);
    resolveMovementThresholdEvent(identifier, builder, field);
    resolveDoubleTapEvent(identifier, builder, field);
    resolveTapEvent(identifier, builder, field);
    resolveLightLevel(identifier, builder, field);
    resolveTemperature(identifier, builder, field);
    resolveBattery(identifier, builder, field);
    resolveButtonClick(identifier, builder, field);
    resolveDoubleClickEvent(identifier, builder, field);
    resolverUtcTime(identifier, builder, field);
    resolveDataLoggerStatus(identifier, builder, field);
    resolveIdentifiedButtonClick(identifier, builder, field);
    resolveHumidity(identifier, builder, field);
    resolveTemperature16Bit(identifier, builder, field);
    resolveBleChannel(identifier, builder, field);
    resolveGpioPinState(identifier, builder, field);
    resolveMovementEvent(identifier, builder, field);
    resolveAirPressure(identifier, builder, field);
    resolvePIRDetection(identifier, builder, field);
    resolveTwoClick(identifier, builder, field);
    resolveAirQuality(identifier, builder, field);
    resolveRoomNumber(identifier, builder, field);
    resolveOccupancy(identifier, builder, field);
    resolveVendorSpecificPayload(identifier, builder, field);
  }

  private void resolveTapEvent(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == TAP_EVENT_IDENTIFIER && field.length >= TAP_EVENT_LENGTH) {
      int tapSeconds = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.lastTap(tapSeconds);
    }
  }

  private void resolveDoubleTapEvent(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == DOUBLE_TAP_IDENTIFIER && field.length >= DOUBLE_TAP_EVENT_LENGTH) {
      int doubleTapSeconds = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.lastDoubleTap(doubleTapSeconds);
    }
  }

  private void resolveMovementThresholdEvent(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == MOVEMENT_THRESHOLD_EVENT_IDENTIFIER && field.length >= MOVEMENT_THRESHOLD_EVENT_LENGTH) {
      int movementSecondsThreshold = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.lastThreshold(movementSecondsThreshold);
    }
  }

  private void resolveRawAccelerometer(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == RAW_ACCELEROMETER_IDENTIFIER && field.length >= RAW_ACCELEROMETER_LENGTH) {
      int rawSensitivity = asInt(field[1]);
      builder.sensitivity(rawSensitivity);
      Acceleration rawAcceleration = new Acceleration(extractSubdata(field, 2, 3));
      builder.acceleration(rawAcceleration);
    }

  }

  private void resolveBasicSystemHealth(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == BASIC_SYSTEM_HEALTH_IDENTIFIER && field.length >= BASIC_SYSTEM_HEALTH_FIELD_LENGTH) {
      final int timestamp = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
      builder.timestamp(timestamp);
      final int batteryLevel = asInt(field[5]);
      builder.batteryLevel(batteryLevel);
    }
  }

  private void resolveAccelerometerData(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == ACCELEROMETER_IDENTIFIER && field.length >= ACCELEROMETER_FIELD_LENGTH) {
      final int sensitivity = asInt(field[1]);
      builder.sensitivity(sensitivity);
      final Acceleration acceleration = new Acceleration(extractSubdata(field, 2, 3));
      builder.acceleration(acceleration);
      final int lastDoubleTap = asIntFromLittleEndianBytes(extractSubdata(field, 5, 2));
      builder.lastDoubleTap(lastDoubleTap);
      final int lastThreshold = asIntFromLittleEndianBytes(extractSubdata(field, 7, 2));
      builder.lastThreshold(lastThreshold);
    }
  }

  private void resolveScanningData(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == SCANNING_IDENTIFIER && field.length >= SCANNING_FIELD_LENGTH) {
      final int bleScans = asInt(field[1]);
      builder.bleScans(bleScans);
      final int wifiScans = asInt(field[2]);
      builder.wifiScans(wifiScans);
      final int bleDevices = asIntFromLittleEndianBytes(extractSubdata(field, 3, 2));
      builder.bleDevices(bleDevices);
    }
  }

  @SuppressWarnings("ConstantConditions")
  private void resolveMoreSystemHealth(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == MORE_SYSTEM_HEALTH_IDENTIFIER && field.length >= MORE_SYSTEM_HEALTH_FIELD_LENGTH) {
      final int timestamp = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
      builder.timestamp(timestamp);
      final int uptime = asIntFromLittleEndianBytes(extractSubdata(field, 5, 2));
      builder.uptime(uptime);
      final int systemLoad = asInt(field[7]);
      builder.systemLoad(systemLoad);
      byte[] bytes = extractSubdata(field, 8, 2);
      final TelemetryError error = TelemetryError.fromValueBytes(bytes);
      builder.error(error);
    }
  }

  private void resolveSensors(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == SENSORS_IDENTIFIER && field.length >= SENSORS_FIELD_LENGTH) {
      final int lightSensor = asInt(field[1]);
      builder.lightSensor(lightSensor);
      final int temperature = field[2];
      builder.temperature(temperature);
    }
  }

  private void resolveLightLevel(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == LIGHT_LEVEL_IDENTIFIER && field.length >= LIGHT_LEVEL_FIELD_LENGTH) {
      final int light = asInt(field[1]);
      builder.lightSensor(light);
    }
  }

  private void resolveTemperature(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == TEMPERATURE_IDENTIFIER && field.length >= TEMPERATURE_FIELD_LENGTH) {
      final int temperature = asInt(field[1]);
      builder.temperature(temperature);
    }
  }

  private void resolveBattery(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == BATTERY_IDENTIFIER && field.length >= BATTERY_FIELD_LENGTH) {
      int battery = asInt(field[1]);
      builder.batteryLevel(battery);
    }
  }

  private void resolveButtonClick(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == BUTTON_CLICK_IDENTIFIER && field.length >= BUTTON_CLICK_FIELD_LENGTH) {
      final int lastSingleClickSeconds = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.lastSingleClick(lastSingleClickSeconds < 0xFFFF ? lastSingleClickSeconds : -1);
    }
  }

  private void resolveDoubleClickEvent(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == BUTTON_DOUBLE_CLICK_IDENTIFIER && field.length >= BUTTON_DOUBLE_CLICK_FIELD_LENGTH) {
      final int lastDoubleClickSeconds = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.lastDoubleClick(lastDoubleClickSeconds);
    }
  }

  private void resolverUtcTime(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == UTC_TIME_IDENTIFIER && field.length >= UTC_TIME_FIELD_LENGTH) {
      int utcTime = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
      builder.timestamp(utcTime);
    }
  }

  private void resolveDataLoggerStatus(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == LOGGING_ENABLED_IDENTIFIER && field.length >= LOGGING_ENABLED_FIELD_LENGTH) {
      final int loggingStatus = asInt(field[1]);
      builder.dataLoggerStatus(loggingStatus == 1 ? DeviceDataLoggerStatus.ENABLED : DeviceDataLoggerStatus.DISABLED);
    }
  }

  private void resolveIdentifiedButtonClick(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == IDENTIFIED_BUTTON_CLICK_IDENTIFIER && field.length >= IDENTIFIED_BUTTON_CLICK_FIELD_LENGTH) {
      final int clickId = asInt(field[1]);
      builder.singleClickCount(clickId);
      final int singleClick = asIntFromLittleEndianBytes(extractSubdata(field, 2, 2));
      builder.lastSingleClick(singleClick < 0xFFFF ? singleClick : -1);
    }
  }

  private void resolveHumidity(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == HUMIDITY_IDENTIFIER && field.length >= HUMIDITY_FIELD_LENGTH) {
      final int humidity = asInt(field[1]);
      builder.humidity(humidity);
    }
  }

  private void resolveTemperature16Bit(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == TEMPERATURE_16_BIT_IDENTIFIER && field.length >= TEMPERATURE_16_BIT_FIELD_LENGTH) {
      final int temperature = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.temperature(temperature / 256.0f);
    }
  }

  private void resolveBleChannel(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if (identifier == BLE_CHANNEL_IDENTIFIER && field.length >= BLE_CHANNEL_FIELD_LENGTH) {
      final int bleChannel = asInt(field[1]);
      builder.bleChannel(bleChannel);
    }
  }

  private void resolveGpioPinState(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == GPIO_PIN_STATE && field.length >= GPIO_PIN_STATE_FIELD_LENGTH) {
      byte definedGpioMask = field[1];
      builder.definedGpioMask(definedGpioMask);
      byte stateGpioMask = field[2];
      builder.stateGpioMask(stateGpioMask);
    }
  }

  private void resolveMovementEvent(int identifier, KontaktTelemetry.Builder builder, byte[] field) {
    if(identifier == MOVEMENT_EVENT_IDENTIFIER && field.length >= MOVEMENT_EVENT_FIELD_LENGTH) {
      int currentMovementId = asInt(field[1]);
      builder.currentMovementId(currentMovementId);
      int threshold = asIntFromLittleEndianBytes(extractSubdata(field, 2, 2));
      builder.lastThreshold(threshold);
    }
  }

  private void resolveAirPressure(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == AIR_PRESSURE_IDENTIFIER && field.length >= AIR_PRESSURE_FIELD_LENGTH) {
      byte[] bytes = extractSubdata(field, 1, 4);
      assert bytes != null;
      float pressure = asFloatFromLittleEndianBytes(bytes);
      builder.airPressure(pressure);
    }
  }

  private void resolvePIRDetection(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == PIR_IDENTIFIER && field.length >= PIR_DETECTION_FIELD_LENGTH) {
      int pirDetectionSeconds = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.pirDetectionSeconds(pirDetectionSeconds < 0xFFFF ? pirDetectionSeconds : -1);
    }
  }

  private void resolveTwoClick(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == TWO_BUTTON_INFO_IDENTIFIER && field.length >= TWO_BUTTON_INFO_FIELD_LENGTH) {
      int twoClickId1 = asInt(field[1]);
      int twoClickId2 = asInt(field[2]);
      int twoClickSeconds = asIntFromLittleEndianBytes(extractSubdata(field, 3, 2));
      builder.singleClickCount(twoClickId1);
      builder.singleClickCount2(twoClickId2);
      builder.lastSingleClick(twoClickSeconds);
    }
  }

  private void resolveAirQuality(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == AIR_QUALITY_IDENTIFIER && field.length >= AIR_QUALITY_FIELD_LENGTH){
      int airQuality = asInt(field[1]);
      builder.airQuality(airQuality);
    }
  }

  private void resolveRoomNumber(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == ROOM_NUMBER_IDENTIFIER && field.length >= ROOM_NUMBER_FIELD_LENGTH) {
      int roomNumber = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
      builder.roomNumber(roomNumber);
    }
  }

  private void resolveOccupancy(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier == OCCUPANCY_IDENTIFIER && field.length >= OCCUPANCY_FIELD_LENGTH){
      int occupancy = asInt(field[1]);
      builder.occupancy(occupancy);
    }
  }

  private void resolveVendorSpecificPayload(int identifier, KontaktTelemetry.Builder builder, byte[] field){
    if(identifier != VENDOR_SPECIFIC_PAYLOAD_IDENTIFIER) return;
    byte[] fieldsData = extractSubdata(field, VENDOR_SPECIFIC_PAYLOAD_OFFSET, field.length - VENDOR_SPECIFIC_PAYLOAD_OFFSET);
    builder.manufacturerId(asIntFromLittleEndianBytes(extractSubdata(field, 1, 2)));
    iterateOverFields(builder, fieldsData);
  }
}
