package com.kontakt.sdk.android.ble.connection;

import android.content.Context;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SecureProfileUtils;

/**
 * Factory class used to obtain {@link KontaktDeviceConnection}, {@link SyncableKontaktDeviceConnection} instances.
 */
public class KontaktDeviceConnectionFactory {

  /**
   * Creates new instance of {@link KontaktDeviceConnection}.
   *
   * @param context {@link Context} instance.
   * @param device {@link RemoteBluetoothDevice} instance.
   * @param connectionListener {@link KontaktDeviceConnection.ConnectionListener} listener instance.
   * @return new connection instance.
   */
  public static KontaktDeviceConnection create(Context context, RemoteBluetoothDevice device,
      KontaktDeviceConnection.ConnectionListener connectionListener) {
    return new KontaktDeviceConnectionImpl(context, device, connectionListener);
  }

  public static KontaktDeviceConnection create(Context context, RemoteBluetoothDevice device, KontaktDeviceConnection.ConnectionListener connectionListener, boolean shouldInitializeCharacteristics){
    return new KontaktDeviceConnectionImpl(context, device, connectionListener, shouldInitializeCharacteristics);
  }

  /**
   * Creates new instance of {@link KontaktDeviceConnection}. Used specifically to connect with Beacon Pro (secure profile) devices.
   *
   * @param context {@link Context} instance.
   * @param secureProfile {@link ISecureProfile} instance.
   * @param connectionListener {@link KontaktDeviceConnection.ConnectionListener} listener instance.
   * @return new connection instance.
   */
  public static KontaktDeviceConnection create(Context context, ISecureProfile secureProfile,
      KontaktDeviceConnection.ConnectionListener connectionListener) {
    return create(context, SecureProfileUtils.asRemoteBluetoothDevice(secureProfile), connectionListener);
  }

  /**
   * Creates new instance of {@link SyncableKontaktDeviceConnection}.
   *
   * @param context {@link Context} instance.
   * @param device {@link RemoteBluetoothDevice} instance.
   * @param connectionListener {@link KontaktDeviceConnection.ConnectionListener} listener instance.
   * @return new connection instance.
   */
  public static SyncableKontaktDeviceConnection createSyncable(Context context, RemoteBluetoothDevice device,
      KontaktDeviceConnection.ConnectionListener connectionListener) {
    return new SyncableKontaktDeviceConnectionImpl(context, device, connectionListener);
  }

  /**
   * Creates new instance of {@link SyncableKontaktDeviceConnection}. Used specifically to connect with Beacon Pro (secure profile) devices.
   *
   * @param context {@link Context} instance.
   * @param secureProfile {@link ISecureProfile} instance.
   * @param connectionListener {@link KontaktDeviceConnection.ConnectionListener} listener instance.
   * @return new connection instance.
   */
  public static SyncableKontaktDeviceConnection createSyncable(Context context, ISecureProfile secureProfile,
      KontaktDeviceConnection.ConnectionListener connectionListener) {
    return createSyncable(context, SecureProfileUtils.asRemoteBluetoothDevice(secureProfile), connectionListener);
  }
}
