package com.kontakt.sdk.android.ble.connection;

import com.kontakt.sdk.android.ble.dfu.FirmwareUpdateListener;
import com.kontakt.sdk.android.ble.security.auth.AuthToken;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.model.Time;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

/**
 * KontaktDeviceConnection connection allows to establish connection with beacon.<br>
 * Since firmware in version 4.0 device does not to have setup password, but below it still needs set password
 * <br><br>
 * All operation with device should be as quick as possible. You should not open connection and wait long time before taking any action.
 * After doing job connection should be closed immediately
 */
public interface KontaktDeviceConnection {

  /**
   * Authorizes with beacon.
   * Authorization is required before certain operations conducted on beacons with firmware 4.0 or higher and beacons pro.<br>
   * These operations are:<br>
   * - {@link #readTime(ReadListener)}<br>
   * - {@link #readLightSensor(ReadListener)}
   *
   * @param token    {@link AuthToken} instance required for authorization.
   * @param callback Authorization result callback.
   */
  void authorize(AuthToken token, AuthorizationCallback callback);

  /**
   * Method for writing secure config. Accessible for device with firmware 4.0 and above and Beacon Pro
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided device's firmware does not support this operation
   * *
   */
  void applySecureConfig(String encryptedConfig, WriteListener writeListener);

  /**
   * Method for writing secure config offline. Accessible for devices with firmware 4.0 or higher and Beacon Pro
   * <br>
   * WARNING: This is an advanced feature that should not be used in normal situations.
   * By applying offline secure configs you accept the fact that the beacon might be desynchronized with Kontakt Cloud.
   * <br><br>
   * Throws {@link com.kontakt.sdk.android.ble.exception.IllegalOperationException} if provided device's firmware does not support this operation
   *
   * @param config        Config that is about to be applied.
   * @param token         {@link AuthToken} token.
   * @param writeListener Operation listener
   */
  void applySecureConfig(Config config, AuthToken token, WriteListener writeListener);

  /**
   * Executes Secure Command.
   *
   * @param secureCommand String representing secure command.
   * @param writeListener Execution listener.
   */
  void executeSecureCommand(String secureCommand, WriteListener writeListener);

  /**
   * Operation that returns beacon's current configuration wrapped in a {@link Config} object.<br>
   * Successful read all operation is equal to successful authorization operation {@link #authorize(AuthToken,
   * AuthorizationCallback)}.<br><br>
   * Only available for beacons with firmware 4.0 or higher and beacons pro.<br>
   *
   * @param token    {@link AuthToken} instance.
   * @param listener Read listener.
   */
  void readAll(AuthToken token, ReadListener<Config> listener);

  /**
   * Operation that returns current UTC time on a device.<br>
   * Only available for beacons pro.<br>
   * Authorization is required before this operation. See {@link #authorize(AuthToken, AuthorizationCallback)}
   *
   * @param listener Read listener
   */
  void readTime(ReadListener<Time> listener);

  /**
   * Operation that synchronizes current UTC time with the beacon.<br>
   * Only available for beacons pro.<br>
   * Authorization is required before this operation. See {@link #authorize(AuthToken, AuthorizationCallback)}
   *
   * @param writeListener Write listener (extra will always be null in this case)
   */
  void syncTime(WriteListener writeListener);

  /**
   * Operation that returns current light sensor reading on a device. Unit: 0 to 100 %<br>
   * Only available for Beacons Pro with firmware 1.7 or higher.<br>
   * Reading light sensor is based on characteristic notification so after using this method a sensor reading will be received continuously every
   * 0.5 of a second.
   * Authorization is required before this operation. See {@link #authorize(AuthToken, AuthorizationCallback)}
   *
   * @param listener Read listener
   */
  void readLightSensor(ReadListener<Integer> listener);

  /**
   * Method for initializing firmware update for Kontakt.io devices with secure profile (for now: Beacon PRO only)
   *
   * @param firmware               {@link Firmware} instance.
   * @param kontaktCloud           {@link KontaktCloud} instance. This is required for Beacon authorization and downloading firmware file.
   * @param firmwareUpdateListener {@link FirmwareUpdateListener} instance. Listener that will report firmware update progress and errors.
   */
  void updateFirmware(Firmware firmware, KontaktCloud kontaktCloud, FirmwareUpdateListener firmwareUpdateListener);

  /**
   * Method for initializing firmware update for Kontakt.io devices with secure profile (for now: Beacon PRO only).
   * This method should be used only uf you have pre-loaded byte array of valid firmware file.
   *
   * @param firmware               {@link Firmware} instance.
   * @param fileBytes              Firmware file bytes.
   * @param kontaktCloud           {@link KontaktCloud} instance. This is required for Beacon authorization.
   * @param firmwareUpdateListener {@link FirmwareUpdateListener} instance. Listener that will report firmware update progress and errors.
   */
  void updateFirmware(Firmware firmware, byte[] fileBytes, KontaktCloud kontaktCloud, FirmwareUpdateListener firmwareUpdateListener);

  /**
   * Gets beacon that the connection is established with.
   *
   * @return the iBeacon device.
   */
  RemoteBluetoothDevice getDevice();

  /**
   * Initializes connection to device provided in the KontaktDeviceConnection constructor.
   *
   * @return true, if the connection attempt was initiated successfully
   */
  boolean connect();

  /**
   * Initializes connection to given device.
   *
   * @param remoteBluetoothDevice device to connect to.
   * @return true, if the connection attempt was initiated successfully.
   */
  boolean connect(RemoteBluetoothDevice remoteBluetoothDevice);

  /**
   * Closes Beacon connection releases provided resources.
   * The method is synchronized.
   */
  void close();

  /**
   * Returns true if Connection with Beacon is established.
   * The method is synchronized.
   *
   * @return the boolean flag indicating whether connection is established.
   */
  boolean isConnected();

  /**
   * Returns true if Beacon was successfully authenticated.
   * The method is synchronized.
   *
   * @return the boolean flag indicating whether beacon was authenticated.
   */
  boolean isAuthenticated();

  /**
   * Returns true if connection is closed.
   *
   * @return the boolean flag indicating whether connection is closed.
   */
  boolean isClosed();

  interface ConnectionListener {

    /**
     * Called when Android Device established connection with beacon.
     */
    void onConnectionOpened();

    /**
     * Called when Beacon device was successfully connected.
     */
    void onConnected();

    /**
     * Called when unexpected error occured during interacting with Beacon device.
     *
     * @param errorCode the error code
     */
    void onErrorOccured(int errorCode);

    /**
     * Called when Android device is disconnected from remote device.
     */
    void onDisconnected();

  }
}
