package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * Preset describes the predefined set of parameters that {@link Device} can be configured with.
 * Contrary to {@link Config}, presets are not removed once the device will be updated with parameters
 * applied from them. They should be interpreted as states in which devices are working.
 * For example, you could define "Long interval profile" where device interval period accepts the
 * the longest interval.
 */
public class Preset implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Parcelable.Creator<Preset> CREATOR = new Parcelable.Creator<Preset>() {
    @Override
    public Preset createFromParcel(Parcel source) {
      return new Preset(source);
    }

    @Override
    public Preset[] newArray(int size) {
      return new Preset[size];
    }
  };

  /**
   * A constant holding the minimum value an interval property can have.
   */
  public static final int INTERVAL_MIN_VALUE = 20;

  /**
   * A constant holding the maximum value an interval property can have.
   */
  public static final int INTERVAL_MAX_VALUE = 10240;

  /**
   * A constant holding the minimum value a txPower property can have.
   */
  public static final int TX_POWER_MIN_VALUE = 0;

  /**
   * A constant holding the maximum value a txPower property can have.
   */
  public static final int TX_POWER_MAX_VALUE = 7;

  private UUID proximity;
  private int interval;
  private int txPower;
  private String name;
  private String description;

  public static Builder builder() {
    return new Builder();
  }

  private Preset() {
    this(new Builder());
  }

  Preset(Builder builder) {
    this.proximity = builder.proximity;
    this.interval = builder.interval;
    this.txPower = builder.txPower;
    this.name = builder.name;
    this.description = builder.description;
  }

  protected Preset(Parcel in) {
    this.proximity = (UUID) in.readSerializable();
    this.interval = in.readInt();
    this.txPower = in.readInt();
    this.name = in.readString();
    this.description = in.readString();
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null || !(obj instanceof Preset)) {
      return false;
    }
    final Preset profile = (Preset) obj;
    return SDKEqualsBuilder.start()
        .equals(proximity, profile.proximity)
        .equals(interval, profile.interval)
        .equals(txPower, profile.txPower)
        .equals(name, profile.name)
        .equals(description, profile.description)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(proximity).append(interval).append(txPower).append(name).append(description).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.proximity);
    dest.writeInt(this.interval);
    dest.writeInt(this.txPower);
    dest.writeString(this.name);
    dest.writeString(this.description);
  }

  public UUID getProximity() {
    return proximity;
  }

  public int getInterval() {
    return interval;
  }

  public int getTxPower() {
    return txPower;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  /**
   * Builder class that is used to build {@link Preset} instances from values configured by the setters.
   */
  public static class Builder {
    UUID proximity;
    int interval = 0;
    int txPower = -1;
    String name;
    String description;

    public Builder proximity(final UUID proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder interval(final int interval) {
      SDKPreconditions.checkState(interval >= INTERVAL_MIN_VALUE && interval <= INTERVAL_MAX_VALUE,
          "interval should be in range from " + INTERVAL_MIN_VALUE + " to " + INTERVAL_MAX_VALUE);
      this.interval = interval;
      return this;
    }

    public Builder txPower(final int txPower) {
      SDKPreconditions.checkState(txPower >= TX_POWER_MIN_VALUE && txPower <= TX_POWER_MAX_VALUE,
          "txPower should be in range from " + TX_POWER_MIN_VALUE + " to " + TX_POWER_MAX_VALUE);
      this.txPower = txPower;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder description(final String description) {
      this.description = description;
      return this;
    }

    public Preset build() {
      return new Preset(this);
    }
  }
}
