package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * An abstract representation of real company, registered in kontakt.io database.
 * <br><br>
 * To create new instance of this class, please use {@link Company.Builder}.
 */
public class Company implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Company> CREATOR = new Creator<Company>() {
    @Override
    public Company createFromParcel(Parcel source) {
      return new Company(source);
    }

    @Override
    public Company[] newArray(int size) {
      return new Company[size];
    }
  };

  private final UUID id;
  private final String name;
  private final String countryCode;
  private final String uniqueId;
  private final List<SubscriptionPlan> subscriptionPlans;

  public static Builder builder() {
    return new Builder();
  }

  Company(Builder builder) {
    this.id = builder.id;
    this.name = builder.name;
    this.countryCode = builder.countryCode;
    this.uniqueId = builder.uniqueId;
    this.subscriptionPlans = builder.subscriptionPlans;
  }

  protected Company(Parcel in) {
    this.id = (UUID) in.readSerializable();
    this.name = in.readString();
    this.countryCode = in.readString();
    this.uniqueId = in.readString();
    this.subscriptionPlans = new ArrayList<>();
    in.readList(this.subscriptionPlans, SubscriptionPlan.class.getClassLoader());
  }

  public UUID getId() {
    return id;
  }

  public String getName() {
    return name;
  }

  public String getCountryCode() {
    return countryCode;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public List<SubscriptionPlan> getSubscriptionPlans() {
    return subscriptionPlans;
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }

    if (obj == null || !(obj instanceof Company)) {
      return false;
    }

    final Company company = (Company) obj;

    return SDKEqualsBuilder.start()
        .equals(id, company.id)
        .equals(countryCode, company.countryCode)
        .equals(name, company.name)
        .equals(uniqueId, company.uniqueId)
        .equals(subscriptionPlans, company.subscriptionPlans)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(id)
        .append(countryCode)
        .append(name)
        .append(uniqueId)
        .append(subscriptionPlans)
        .build();
  }

  @Override
  public String toString() {
    return "Company{"
        + "id=" + id
        + ", name='" + name + '\''
        + ", countryCode='" + countryCode + '\''
        + ", uniqueId='" + uniqueId + '\''
        + ", subscriptionPlans=" + subscriptionPlans
        + '}';
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.id);
    dest.writeString(this.name);
    dest.writeString(this.countryCode);
    dest.writeString(this.uniqueId);
    dest.writeList(this.subscriptionPlans);
  }

  /**
   * Builder class that is used to build {@link Company} instances from values configured by the setters.
   */
  public static class Builder {
    UUID id;
    String name;
    String countryCode;
    String uniqueId;
    List<SubscriptionPlan> subscriptionPlans = new ArrayList<>();

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder countryCode(final String countryCode) {
      this.countryCode = countryCode;
      return this;
    }

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder subscriptionPlans(final List<SubscriptionPlan> subscriptionPlans) {
      this.subscriptionPlans = subscriptionPlans;
      return this;
    }

    public Company build() {
      return new Company(this);
    }
  }
}
