package com.kontakt.sdk.android.cloud.api.executor.places;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.PlacesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.PlacesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Place;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

/**
 * Request executor provided by {@link PlacesApi}. Use this class if you want to update
 * places through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   // define updated place object
 *   kontaktCloud.places().update(ID)
 *      .with(updatedPlace)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that place must be specified so invocation of {@code with} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class UpdatePlaceRequestExecutor extends RequestExecutor<String> {

  private final PlacesService placesService;
  private final UUID id;
  private Place place;

  /**
   * Constructs request executor initialized with corresponding service class and place ID.
   *
   * @param placesService the places API service.
   * @param id the place's identifier.
   */
  public UpdatePlaceRequestExecutor(PlacesService placesService, UUID id) {
    this.placesService = placesService;
    this.id = id;
  }

  /**
   * Specifies place data to update. The method invocation is obligatory while using this request executor.
   *
   * @param place updated place data.
   * @return this request executor.
   */
  public UpdatePlaceRequestExecutor with(final Place place) {
    checkNotNull(place, "place cannot be null");
    this.place = place;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return placesService.updatePlace(params());
  }

  private void checkPreconditions() {
    checkState(place != null, "cannot update place - specify place data to update");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Places.PLACE_ID_PARAMETER, id.toString());
    if (place.getName() != null) {
      params.put(CloudConstants.Places.NAME_PARAMETER, place.getName());
    }
    if (place.getScale() != 0.0d) {
      params.put(CloudConstants.Places.SCALE_PARAMETER, String.valueOf(place.getScale()));
    }
    if (place.getCoordinates() != null) {
      params.put(CloudConstants.Places.COORDINATES_PARAMETER, place.getCoordinates().toString());
    }
    if (place.getSchema() != null) {
      params.put(CloudConstants.Places.SCHEMA_PARAMETER, place.getSchema());
    }
    return params;
  }
}
