package com.kontakt.sdk.android.cloud.api.executor.places;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.PlacesApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.PlacesService;
import com.kontakt.sdk.android.cloud.response.paginated.Places;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Request executor provided by {@link PlacesApi}. Use this class if you want to obtain
 * places through fluent API in chained fashion, for instance:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Places places = kontaktCloud.places().fetch()
 *      .fromVenues(venueIds)
 *      .maxResult(20)
 *      .execute();
 *   </code>
 * </pre>
 */
public class PlacesRequestExecutor extends PaginatedRequestExecutor<Places> {

  private final PlacesService placesService;

  private UUID[] placeIds;
  private UUID[] venueIds;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param placesService the places API service.
   */
  public PlacesRequestExecutor(PlacesService placesService) {
    this.placesService = placesService;
  }

  /**
   * Specifies places identifiers. Use that method if you want to fetch particular place by ID.
   *
   * @param ids places identifiers.
   * @return this request executor.
   */
  public PlacesRequestExecutor withIds(final UUID... ids) {
    this.placeIds = checkNotNull(ids, "Place IDs cannot be null");
    return this;
  }

  /**
   * Specifies places identifiers. Use that method if you want to fetch particular place by ID.
   *
   * @param ids places identifiers.
   * @return this request executor.
   */
  public PlacesRequestExecutor withIds(final List<UUID> ids) {
    checkNotNull(ids, "Place IDs cannot be null");
    final int size = ids.size();
    this.placeIds = ids.toArray(new UUID[size]);
    return this;
  }

  /**
   * Specifies venues for whose places belong to.
   *
   * @param venueIds venues identifiers.
   * @return this request executor.
   */
  public PlacesRequestExecutor fromVenues(final UUID... venueIds) {
    this.venueIds = checkNotNull(venueIds, "Venue IDs cannot be null");
    return this;
  }

  /**
   * Specifies venues for whose places belong to.
   *
   * @param venueIds venues identifiers.
   * @return this request executor.
   */
  public PlacesRequestExecutor fromVenues(final List<UUID> venueIds) {
    checkNotNull(venueIds, "Venue IDs cannot be null");
    final int size = venueIds.size();
    this.venueIds = venueIds.toArray(new UUID[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PlacesRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PlacesRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PlacesRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PlacesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PlacesRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Places> prepareCall() {
    final Call<Places> call;
    if (eTag != null) {
      call = placesService.getPlaces(params(), eTag);
    } else {
      call = placesService.getPlaces(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (placeIds != null && placeIds.length > 0) {
      params.put(CloudConstants.Places.PLACE_ID_PARAMETER, StringUtils.join(placeIds, ","));
    }
    if (venueIds != null && venueIds.length > 0) {
      params.put(CloudConstants.Places.VENUE_ID_PARAMETER, StringUtils.join(venueIds, ","));
    }
    return params;
  }
}
