package com.kontakt.sdk.android.cloud.api.executor.actions;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActionsApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.paginated.Actions;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link ActionsApi}. Use this class if you want to obtain actions
 * through fluent API in chained fashion. Here is an example of how to use this executor:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Actions actions = kontaktCloud.actions().fetch()
 *      .forDevices("AxT7")
 *      .startIndex(4)
 *      .maxResult(10)
 *      .filter("")
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that devices must be specified so invocation of {@code forDevices} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class ActionsRequestExecutor extends PaginatedRequestExecutor<Actions> {

  private final ActionsService actionsService;
  private String[] uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param actionsService the actions API service.
   */
  public ActionsRequestExecutor(final ActionsService actionsService) {
    this.actionsService = actionsService;
  }

  /**
   * Specifies associated devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public ActionsRequestExecutor forDevices(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies associated devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public ActionsRequestExecutor forDevices(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActionsRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActionsRequestExecutor maxResult(int maxResult) {
    super.startIndex(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActionsRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActionsRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActionsRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Actions execute() throws IOException, KontaktCloudException {
    SDKPreconditions.checkState(uniqueIds != null, "Cannot get actions - specify devices");
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<Actions> callback) {
    SDKPreconditions.checkState(uniqueIds != null, "Cannot get actions - specify devices");
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Actions> prepareCall() {
    final Call<Actions> call;
    if (eTag != null) {
      call = actionsService.getActions(params(), eTag);
    } else {
      call = actionsService.getActions(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (uniqueIds != null) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    }
    return params;
  }
}
