package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.places.CreatePlaceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.places.DeletePlaceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.places.PlaceSchemaImageRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.places.PlacesRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.places.UpdatePlaceRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.PlacesService;
import com.kontakt.sdk.android.common.model.Place;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * An abstract representation of Place API endpoint. Provides executors that can be
 * used to handle all place-related requests.
 */
public class PlacesApi {

  private final PlacesService placesService;

  /**
   * Constructs Place API endpoint initialized with corresponding service class.
   * @param placesService the places API service.
   */
  public PlacesApi(PlacesService placesService) {
    this.placesService = placesService;
  }

  /**
   * Provides executor that can be used to obtain places.
   *
   * @return the request executor.
   */
  public PlacesRequestExecutor fetch() {
    return new PlacesRequestExecutor(placesService);
  }

  /**
   * Provides executor that can be used to load place's schema image file.
   *
   * @param id the place's identifier
   * @return the request executor.
   */
  public PlaceSchemaImageRequestExecutor schemaImageOf(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new PlaceSchemaImageRequestExecutor(placesService, id);
  }

  /**
   * Provides executor that can be used to create new place.
   *
   * @param place the place.
   * @return the request executor.
   */
  public CreatePlaceRequestExecutor create(final Place place) {
    checkNotNull(place, "place cannot be null");
    return new CreatePlaceRequestExecutor(placesService, place);
  }

  /**
   * Provides executor that can be used to update a place.
   *
   * @param id the place's identifier.
   * @return the request executor.
   */
  public UpdatePlaceRequestExecutor update(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new UpdatePlaceRequestExecutor(placesService, id);
  }

  /**
   * Provides executor that can be used to delete a venue.
   *
   * @param id the venue's unique identifier.
   * @return the request executor.
   */
  public DeletePlaceRequestExecutor delete(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new DeletePlaceRequestExecutor(placesService, id);
  }
}
