package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.activities.ActivitiesRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.activities.CreateActivityRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.activities.DeleteActivityRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.activities.UpdateActivityRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActivitiesService;
import com.kontakt.sdk.android.common.model.Activity;
import java.util.UUID;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * An abstract representation of Activities API endpoint. Provides executors that can be
 * used to handle all activity-related requests.
 */
public class ActivitiesApi {

  private final ActivitiesService activitiesService;

  /**
   * Constructs Activities API endpoint initialized with corresponding service class.
   *
   * @param activitiesService the triggers API service.
   */
  public ActivitiesApi(final ActivitiesService activitiesService) {
    this.activitiesService = activitiesService;
  }

  /**
   * Provides executor that can be used to obtain activities.
   *
   * @return the request executor.
   */
  public ActivitiesRequestExecutor fetch() {
    return new ActivitiesRequestExecutor(activitiesService);
  }

  /**
   * Provides executor that can be used to create new activity.
   *
   * @param activity the activity instance.
   * @return the request executor.
   */
  public CreateActivityRequestExecutor create(final Activity activity) {
    checkNotNull(activity, "Activity cannot be null");
    return new CreateActivityRequestExecutor(activitiesService, activity);
  }

  /**
   * Provides executor that can be used to update an activity.
   *
   * @param id the activity's unique identifier.
   * @return the request executor.
   */
  public UpdateActivityRequestExecutor update(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new UpdateActivityRequestExecutor(activitiesService, id);
  }

  /**
   * Provides executor that can be used to delete a activity.
   *
   * @param id the activity's unique identifier.
   * @return the request executor.
   */
  public DeleteActivityRequestExecutor delete(final UUID id) {
    checkNotNull(id, "ID cannot be null");
    return new DeleteActivityRequestExecutor(activitiesService, id);
  }
}
