package com.kontakt.sdk.android.cloud.adapter;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.PacketType;
import com.kontakt.sdk.android.common.model.PowerSaving;
import com.kontakt.sdk.android.common.model.PowerSavingFeature;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * Converts {@link Config} objects to and from JSON.
 */
public class ConfigTypeAdapter extends BaseTypeAdapter<Config> {

  private final Gson gson = new Gson();

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(JsonWriter out, Config config) throws IOException {
    writeUUID(out, CloudConstants.Configs.PROXIMITY_PARAMETER, config.getProximity());
    writeInteger(out, CloudConstants.Configs.MAJOR_PARAMETER, config.getMajor());
    writeInteger(out, CloudConstants.Configs.MINOR_PARAMETER, config.getMinor());
    writeInteger(out, CloudConstants.Configs.TX_POWER_PARAMETER, config.getTxPower());
    writeInteger(out, CloudConstants.Configs.INTERVAL_PARAMETER, config.getInterval());
    writeString(out, CloudConstants.Configs.NAMESPACE_PARAMETER, config.getNamespace());
    writeString(out, CloudConstants.Configs.INSTANCE_ID_PARAMETER, config.getInstanceId());
    final String url = config.getUrl();
    if (url != null) {
      final String encodedUrl = EddystoneUtils.toHexString(EddystoneUtils.serializeUrl(url));
      writeString(out, CloudConstants.Configs.URL_PARAMETER, encodedUrl);
    }
    writeProfiles(out, config.getProfiles());
    writePackets(out, config.getPackets());
    writeBoolean(out, CloudConstants.Configs.SHUFFLED_PARAMETER, config.isShuffled());
    writeString(out, CloudConstants.Configs.NAME_PARAMETER, config.getName());
    writeString(out, CloudConstants.Configs.PASSWORD_PARAMETER, config.getPassword());
    writeRssiCalibration(out, config.getRssi1m(), CloudConstants.Configs.RSSI_1M_PARAMETER);
    writeRssiCalibration(out, config.getRssi0m(), CloudConstants.Configs.RSSI_0M_PARAMETER);

    // write secure config parameters
    writeString(out, CloudConstants.Configs.SECURE_REQUEST_PARAMETER, config.getSecureRequest());
    writeString(out, CloudConstants.Configs.SECURE_RESPONSE_PARAMETER, config.getSecureResponse());
    writeLong(out, CloudConstants.Configs.SECURE_RESPONSE_TIME_PARAMETER, config.getSecureResponseTime());

    // write power saving features
    writePowerSaving(out, config);

    // write kontakt telemetry parameters
    writeInteger(out, CloudConstants.Configs.TEMPERATURE_OFFSET_PARAMETER, config.getTemperatureOffset());

    // write custom fields
    writeCustomConfiguration(out, config.getCustomConfiguration());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Config read(JsonReader in) throws IOException {
    final Config.Builder configBuilder = new Config.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Common.UNIQUE_ID_PARAMETER:
          final String uniqueId = in.nextString();
          configBuilder.uniqueId(uniqueId);
          break;
        case CloudConstants.Configs.PROXIMITY_PARAMETER:
          final UUID proximity = UUID.fromString(in.nextString());
          configBuilder.proximity(proximity);
          break;
        case CloudConstants.Configs.MAJOR_PARAMETER:
          final int major = in.nextInt();
          configBuilder.major(major);
          break;
        case CloudConstants.Configs.MINOR_PARAMETER:
          final int minor = in.nextInt();
          configBuilder.minor(minor);
          break;
        case CloudConstants.Configs.TX_POWER_PARAMETER:
          final int txPower = in.nextInt();
          configBuilder.txPower(txPower);
          break;
        case CloudConstants.Configs.INTERVAL_PARAMETER:
          final int interval = in.nextInt();
          configBuilder.interval(interval);
          break;
        case CloudConstants.Configs.NAMESPACE_PARAMETER:
          final String namespace = in.nextString();
          configBuilder.namespace(namespace);
          break;
        case CloudConstants.Configs.URL_PARAMETER:
          final String url = in.nextString();
          configBuilder.url(url);
          break;
        case CloudConstants.Configs.INSTANCE_ID_PARAMETER:
          final String instanceId = in.nextString();
          configBuilder.instanceId(instanceId);
          break;
        case CloudConstants.Configs.PROFILES_PARAMETER:
          final List<DeviceProfile> profiles = readProfiles(in);
          configBuilder.profiles(profiles);
          break;
        case CloudConstants.Configs.PACKETS_PARAMETER:
          final List<PacketType> packets = readPackets(in);
          configBuilder.packets(packets);
          break;
        case CloudConstants.Configs.SHUFFLED_PARAMETER:
          final boolean shuffled = in.nextBoolean();
          configBuilder.shuffled(shuffled);
          break;
        case CloudConstants.Configs.NAME_PARAMETER:
          final String name = in.nextString();
          configBuilder.name(name);
          break;
        case CloudConstants.Configs.PASSWORD_PARAMETER:
          final String password = in.nextString();
          configBuilder.password(password);
          break;
        case CloudConstants.Configs.RSSI_1M_PARAMETER:
          configBuilder.rssi1m(readRssiCalibration(in));
          break;
        case CloudConstants.Configs.RSSI_0M_PARAMETER:
          configBuilder.rssi0m(readRssiCalibration(in));
          break;
        case CloudConstants.Configs.SECURE_REQUEST_PARAMETER:
          final String secureRequest = in.nextString();
          configBuilder.secureRequest(secureRequest);
          break;
        case CloudConstants.Configs.POWER_SAVING_PREFIX:
          final PowerSaving powerSaving = readPowerSaving(in);
          configBuilder.powerSaving(powerSaving);
          break;
        case CloudConstants.Configs.TEMPERATURE_OFFSET_PARAMETER:
          final int temperatureOffset = in.nextInt();
          configBuilder.temperatureOffset(temperatureOffset);
          break;
        case CloudConstants.Configs.CUSTOM_CONFIGURATION_PARAMETER:
          try {
            final Map<String, String> customConfiguration = readCustomConfiguration(in);
            configBuilder.customConfiguration(customConfiguration);
          } catch (Exception e) {
            in.skipValue();
          }
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    return configBuilder.build();
  }

  private void writePowerSaving(JsonWriter out, Config config) throws IOException {
    PowerSaving powerSaving = config.getPowerSaving();
    if (powerSaving == null) {
      return;
    }
    out.name(CloudConstants.Configs.POWER_SAVING_PREFIX);
    out.beginObject();
    writeLong(out, CloudConstants.Configs.MOVE_SUSPEND_TIMEOUT_PARAMETER, powerSaving.getMoveSuspendTimeout());
    writeInteger(out, CloudConstants.Configs.LIGHT_SENSOR_HYSTERESIS_PARAMETER, powerSaving.getLightSensorHysteresis());
    writeInteger(out, CloudConstants.Configs.LIGHT_SENSOR_THRESHOLD_PARAMETER, powerSaving.getLightSensorThreshold());
    writeLong(out, CloudConstants.Configs.LIGHT_SENSOR_SAMPLING_INTERVAL_PARAMETER, powerSaving.getLightSensorSamplingInterval());
    writeFeatures(out, powerSaving.getFeatures());
    out.endObject();
  }

  private void writeProfiles(JsonWriter out, List<DeviceProfile> profiles) throws IOException {
    out.name(CloudConstants.Configs.PROFILES_PARAMETER);
    if (profiles == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (DeviceProfile profile : profiles) {
      out.value(profile.name());
    }
    out.endArray();
  }

  private void writePackets(JsonWriter out, List<PacketType> packets) throws IOException {
    out.name(CloudConstants.Configs.PACKETS_PARAMETER);
    if (packets == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (PacketType packet : packets) {
      out.value(packet.name());
    }
    out.endArray();
  }

  private void writeFeatures(JsonWriter out, List<PowerSavingFeature> features) throws IOException {
    out.name(CloudConstants.Configs.FEATURES_PARAMETER);
    if (features == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (PowerSavingFeature feature : features) {
      out.value(feature.toString());
    }
    out.endArray();
  }

  private void writeRssiCalibration(JsonWriter out, List<Integer> rssi, String cloudParam) throws IOException {
    out.name(cloudParam);
    if (rssi == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (Integer i : rssi) {
      out.value(i);
    }
    out.endArray();
  }

  private void writeCustomConfiguration(JsonWriter out, Map<String, String> customConfiguration) throws IOException {
    out.name(CloudConstants.Configs.CUSTOM_CONFIGURATION_PARAMETER);
    if (customConfiguration == null) {
      out.nullValue();
      return;
    }
    out.value(gson.toJson(customConfiguration));
  }

  PowerSaving readPowerSaving(JsonReader in) throws IOException {
    PowerSaving.Builder powerSavingBuilder = new PowerSaving.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }

      switch (fieldName) {
        case CloudConstants.Configs.MOVE_SUSPEND_TIMEOUT_PARAMETER:
          powerSavingBuilder.moveSuspendTimeout(in.nextLong());
          break;
        case CloudConstants.Configs.LIGHT_SENSOR_HYSTERESIS_PARAMETER:
          powerSavingBuilder.lightSensorHysteresis(in.nextInt());
          break;
        case CloudConstants.Configs.LIGHT_SENSOR_SAMPLING_INTERVAL_PARAMETER:
          powerSavingBuilder.lightSensorSamplingInterval(in.nextInt());
          break;
        case CloudConstants.Configs.LIGHT_SENSOR_THRESHOLD_PARAMETER:
          powerSavingBuilder.lightSensorThreshold(in.nextInt());
          break;
        case CloudConstants.Configs.FEATURES_PARAMETER:
          powerSavingBuilder.features(readFeatures(in));
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    return powerSavingBuilder.build();
  }

  List<DeviceProfile> readProfiles(JsonReader in) throws IOException {
    in.beginArray();
    List<DeviceProfile> profiles = new ArrayList<>();
    while (in.hasNext()) {
      profiles.add(DeviceProfile.valueOf(in.nextString()));
    }
    in.endArray();
    return profiles;
  }

  List<PacketType> readPackets(JsonReader in) throws IOException {
    in.beginArray();
    List<PacketType> packets = new ArrayList<>();
    while (in.hasNext()) {
      packets.add(PacketType.valueOf(in.nextString()));
    }
    in.endArray();
    return packets;
  }

  List<PowerSavingFeature> readFeatures(JsonReader in) throws IOException {
    in.beginArray();
    List<PowerSavingFeature> features = new ArrayList<>();
    while (in.hasNext()) {
      PowerSavingFeature f = PowerSavingFeature.fromString(in.nextString());
      if (f != null) {
        features.add(f);
      }
    }
    in.endArray();
    return features;
  }

  List<Integer> readRssiCalibration(JsonReader in) throws IOException {
    in.beginArray();
    List<Integer> rssi = new ArrayList<>();
    while (in.hasNext()) {
      rssi.add(in.nextInt());
    }
    in.endArray();
    return rssi;
  }

  Map<String, String> readCustomConfiguration(JsonReader in) throws IOException {
    in.beginObject();
    Map<String, String> customConfiguration = new HashMap<>();
    while (in.hasNext()) {
      customConfiguration.put(in.nextName(), in.nextString());
    }
    in.endObject();
    return customConfiguration;
  }
}
