package com.kontakt.sdk.android.ble.spec;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * Telemetry class that provides information about system health for battery operated devices, aggregates accelerometer data,
 * sensors and scanning.
 *
 * Scans per second encoding:
 * 0      |   Exactly zero scans over last 10 seconds
 * 0-249  |   Rounded number of scans/s.
 * 250    |   250 or more scans/s
 * 251    |   0.1 - 0.2 scans/s
 * 252    |   0.3 - 0.4 scans/s
 * 253    |   0.5 - 0.6 scans/s
 * 254    |   0.7 - 0.8 scans/s
 */
public class KontaktTelemetry implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Parcelable.Creator<KontaktTelemetry> CREATOR = new Parcelable.Creator<KontaktTelemetry>() {
    @Override
    public KontaktTelemetry createFromParcel(Parcel source) {
      return new KontaktTelemetry(source);
    }

    @Override
    public KontaktTelemetry[] newArray(int size) {
      return new KontaktTelemetry[size];
    }
  };

  public static final int BATTERY_LEVEL_EXTERNAL_POWER_SUPPLIED_VALUE = 255;
  public static final int DOUBLE_TAP_DISABLED_VALUE = 65535;
  public static final int THRESHOLD_DISABLED_VALUE = 65535;
  public static final int LOAD_AVERAGE_NOT_SUPPORTED_VALUE = 255;
  public static final int BLE_SCANS_NOT_SUPPORTED_VALUE = 255;
  public static final int WIFI_SCANS_NOT_SUPPORTED_VALUE = 255;
  public static final int BLE_DEVICES_NOT_SUPPORTED_VALUE = 65535;

  // Basic System Health
  private final int timestamp;
  private final int batteryLevel;

  // More System Health
  private final int uptime;
  private final int systemLoad;
  private final TelemetryError error;

  // Accelerometer data
  private final int sensitivity;
  private final Acceleration acceleration;
  private final int lastDoubleTap;
  private final int lastThreshold;

  // Sensors
  private final int lightSensor;
  private final int temperature;

  // Scanning
  private final int bleScans;
  private final int wifiScans;
  private final int bleDevices;

  // Button click
  private final int lastSingleClick;
  private final int singleClickCount;

  // Data Logger
  private final DeviceDataLoggerStatus dataLoggerStatus;

  // Humidity
  private final int humidity;

  // BLE Channel
  private final int bleChannel;

  private KontaktTelemetry(final Builder builder) {
    this.timestamp = builder.timestamp;
    this.batteryLevel = builder.batteryLevel;
    this.uptime = builder.uptime;
    this.systemLoad = builder.systemLoad;
    this.error = builder.error;
    this.sensitivity = builder.sensitivity;
    this.acceleration = builder.acceleration;
    this.lastDoubleTap = builder.lastDoubleTap;
    this.lastThreshold = builder.lastThreshold;
    this.lightSensor = builder.lightSensor;
    this.temperature = builder.temperature;
    this.bleScans = builder.bleScans;
    this.wifiScans = builder.wifiScans;
    this.bleDevices = builder.bleDevices;
    this.lastSingleClick = builder.lastSingleClick;
    this.singleClickCount = builder.singleClickCount;
    this.dataLoggerStatus = builder.dataLoggerStatus;
    this.humidity = builder.humidity;
    this.bleChannel = builder.bleChannel;
  }

  protected KontaktTelemetry(Parcel in) {
    this.timestamp = in.readInt();
    this.batteryLevel = in.readInt();
    this.uptime = in.readInt();
    this.systemLoad = in.readInt();
    int tmpError = in.readInt();
    this.error = tmpError == -1 ? null : TelemetryError.values()[tmpError];
    this.sensitivity = in.readInt();
    this.acceleration = in.readParcelable(Acceleration.class.getClassLoader());
    this.lastDoubleTap = in.readInt();
    this.lastThreshold = in.readInt();
    this.lightSensor = in.readInt();
    this.temperature = in.readInt();
    this.bleScans = in.readInt();
    this.wifiScans = in.readInt();
    this.bleDevices = in.readInt();
    this.lastSingleClick = in.readInt();
    this.singleClickCount = in.readInt();
    this.dataLoggerStatus = DeviceDataLoggerStatus.valueOf(in.readString());
    this.humidity = in.readInt();
    this.bleChannel = in.readInt();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    KontaktTelemetry that = (KontaktTelemetry) o;

    if (timestamp != that.timestamp) return false;
    if (batteryLevel != that.batteryLevel) return false;
    if (uptime != that.uptime) return false;
    if (systemLoad != that.systemLoad) return false;
    if (sensitivity != that.sensitivity) return false;
    if (lastDoubleTap != that.lastDoubleTap) return false;
    if (lastThreshold != that.lastThreshold) return false;
    if (lightSensor != that.lightSensor) return false;
    if (temperature != that.temperature) return false;
    if (bleScans != that.bleScans) return false;
    if (wifiScans != that.wifiScans) return false;
    if (bleDevices != that.bleDevices) return false;
    if (error != that.error) return false;
    if (lastSingleClick != that.lastSingleClick) return false;
    if (singleClickCount != that.singleClickCount) return false;
    if (dataLoggerStatus != that.dataLoggerStatus) return false;
    if (humidity != that.humidity) return false;
    if (bleChannel != that.bleChannel) return false;
    return acceleration != null ? acceleration.equals(that.acceleration) : that.acceleration == null;
  }

  @Override
  public int hashCode() {
    int result = timestamp;
    result = 31 * result + batteryLevel;
    result = 31 * result + uptime;
    result = 31 * result + systemLoad;
    result = 31 * result + (error != null ? error.hashCode() : 0);
    result = 31 * result + sensitivity;
    result = 31 * result + (acceleration != null ? acceleration.hashCode() : 0);
    result = 31 * result + lastDoubleTap;
    result = 31 * result + lastThreshold;
    result = 31 * result + lightSensor;
    result = 31 * result + temperature;
    result = 31 * result + bleScans;
    result = 31 * result + wifiScans;
    result = 31 * result + bleDevices;
    result = 31 * result + lastSingleClick;
    result = 31 * result + singleClickCount;
    result = 31 * result + dataLoggerStatus.ordinal();
    result = 31 * result + humidity;
    result = 31 * result + bleChannel;
    return result;
  }

  @Override
  public String toString() {
    return "KontaktTelemetry{"
            + "timestamp="
            + timestamp
            + ", batteryLevel="
            + batteryLevel
            + ", uptime="
            + uptime
            + ", systemLoad="
            + systemLoad
            + ", error="
            + error
            + ", sensitivity="
            + sensitivity
            + ", acceleration="
            + acceleration
            + ", lastDoubleTap="
            + lastDoubleTap
            + ", lastThreshold="
            + lastThreshold
            + ", lightSensor="
            + lightSensor
            + ", temperature="
            + temperature
            + ", bleScans="
            + bleScans
            + ", wifiScans="
            + wifiScans
            + ", bleDevices="
            + bleDevices
            + ", lastSingleClick="
            + lastSingleClick
            + ", singleClickCount="
            + singleClickCount
            + ", dataLoggerStatus="
            + dataLoggerStatus.name()
            + ", humidity="
            + humidity
            + ", bleChannel="
            + bleChannel
            + '}';
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.timestamp);
    dest.writeInt(this.batteryLevel);
    dest.writeInt(this.uptime);
    dest.writeInt(this.systemLoad);
    dest.writeInt(this.error == null ? -1 : this.error.ordinal());
    dest.writeInt(this.sensitivity);
    dest.writeParcelable(this.acceleration, flags);
    dest.writeInt(this.lastDoubleTap);
    dest.writeInt(this.lastThreshold);
    dest.writeInt(this.lightSensor);
    dest.writeInt(this.temperature);
    dest.writeInt(this.bleScans);
    dest.writeInt(this.wifiScans);
    dest.writeInt(this.bleDevices);
    dest.writeInt(this.lastSingleClick);
    dest.writeInt(this.singleClickCount);
    dest.writeString(this.dataLoggerStatus.name());
    dest.writeInt(this.humidity);
    dest.writeInt(this.bleChannel);
  }

  /**
   * Returns Unix timestamp (seconds since 1970).
   *
   * @return the timestamp.
   */
  public int getTimestamp() {
    return timestamp;
  }

  /**
   * Returns battery level percentage (0 - 100).
   *
   * @return the battery level.
   */
  public int getBatteryLevel() {
    return batteryLevel;
  }

  /**
   * Returns uptime rounded to hours.
   *
   * @return the uptime.
   */
  public int getUptime() {
    return uptime;
  }

  /**
   * Returns percentage of system load average over 15 min.
   *
   * @return the system load percentage.
   */
  public int getSystemLoad() {
    return systemLoad;
  }

  /**
   * Returns error flags.
   *
   * @return the error.
   */
  public TelemetryError getError() {
    return error;
  }

  /**
   * Returns accelerometer sensitivity. Unit: mg/digit.
   *
   * @return the accelerometer sensitivity.
   */
  public int getSensitivity() {
    return sensitivity;
  }

  /**
   * Returns raw accelerometer data filtered with lowpass filter.
   *
   * @return the accelerometer data.
   */
  public Acceleration getAcceleration() {
    return acceleration;
  }

  /**
   * Returns seconds since last doubletap event. Doubletap event occurs when doubletap pattern is detected by accelerometer.
   * Saturates on 65535 (0xFFFF) value. 65535 (0xFFFF) is default value when Doubletap is disabled.
   *
   * @return the seconds since last doubletap event.
   */
  public int getLastDoubleTap() {
    return lastDoubleTap;
  }

  /**
   * Returns seconds since last threshold event. Threshold event occurs when acceleration exceeds configured threshold.
   * Saturates on 65535 (0xFFFF) value. 65535 (0xFFFF) is default value when Threshold is disabled.
   *
   * @return the seconds since last threshold event.
   */
  public int getLastThreshold() {
    return lastThreshold;
  }

  /**
   * Returns light sensor percentage (0 - 100).
   *
   * @return the light sensor percentage.
   */
  public int getLightSensor() {
    return lightSensor;
  }

  /**
   * Returns temperature in Celsius degrees.
   *
   * @return the temperature value.
   */
  public int getTemperature() {
    return temperature;
  }

  /**
   * Returns BLE scanning statistics (10 seconds average).
   *
   * @return the BLE scanning statistics.
   */
  public int getBleScans() {
    return bleScans;
  }

  /**
   * Returns WiFi scanning statistics (10 seconds average).
   *
   * @return the WiFi scanning statistics.
   */
  public int getWifiScans() {
    return wifiScans;
  }

  /**
   * Returns number of scanned BLE devices (10 seconds average).
   *
   * @return the scanned BLE devices amount.
   */
  public int getBleDevices() {
    return bleDevices;
  }

  /**
   * Returns seconds since last click event. Click event is specified in Button Specification Behavior.
   * Saturates on 65535 (0xFFFF) value. 65535 (0xFFFF) default value when button click is disabled.
   *
   * @return seconds since last click.
   */
  public int getLastSingleClick() {
    return lastSingleClick;
  }

  /**
   * Returns a number of click events. Click event is specified in Button Specification Behavior.
   *
   * @return the number of click events.
   */
  public int getSingleClickCount() {
    return singleClickCount;
  }

  /**
   * A value indicating whether the data logger functionality is turned `ON`.
   *
   * @return data logger enumeration status.
   */
  public DeviceDataLoggerStatus getDataLoggerStatus() {
    return dataLoggerStatus;
  }

  /**
   * Returns Relative Humidity Percentage (0-100).
   *
   * @return the humidity percentage value.
   */
  public int getHumidity() {
    return humidity;
  }

  /**
   * Returns BLE Channel value.
   *
   * @return the BLE channel value.
   */
  public int getBleChannel() {
    return bleChannel;
  }

  public static class Builder {
    // Basic System Health
    private int timestamp;
    private int batteryLevel;

    // More System Health
    private int uptime;
    private int systemLoad;
    private TelemetryError error;

    // Accelerometer data
    private int sensitivity;
    private Acceleration acceleration;
    private int lastDoubleTap;
    private int lastThreshold;

    // Sensors
    private int lightSensor;
    private int temperature;

    // Scanning
    private int bleScans;
    private int wifiScans;
    private int bleDevices;

    // Button
    private int lastSingleClick;
    private int singleClickCount;

    // Data Logger
    private DeviceDataLoggerStatus dataLoggerStatus = DeviceDataLoggerStatus.UNAVAILABLE;

    // Humidity
    private int humidity;

    // BLE Channel
    private int bleChannel;

    public Builder timestamp(final int timestamp) {
      this.timestamp = timestamp;
      return this;
    }

    public Builder batteryLevel(final int batteryLevel) {
      this.batteryLevel = batteryLevel;
      return this;
    }

    public Builder uptime(final int uptime) {
      this.uptime = uptime;
      return this;
    }

    public Builder systemLoad(final int systemLoad) {
      this.systemLoad = systemLoad;
      return this;
    }

    public Builder error(final TelemetryError error) {
      this.error = error;
      return this;
    }

    public Builder sensitivity(final int sensitivity) {
      this.sensitivity = sensitivity;
      return this;
    }

    public Builder acceleration(final Acceleration acceleration) {
      this.acceleration = acceleration;
      return this;
    }

    public Builder lastDoubleTap(final int lastDoubleTap) {
      this.lastDoubleTap = lastDoubleTap;
      return this;
    }

    public Builder lastThreshold(final int lastThreshold) {
      this.lastThreshold = lastThreshold;
      return this;
    }

    public Builder lightSensor(final int lightSensor) {
      this.lightSensor = lightSensor;
      return this;
    }

    public Builder temperature(final int temperature) {
      this.temperature = temperature;
      return this;
    }

    public Builder bleScans(final int bleScans) {
      this.bleScans = bleScans;
      return this;
    }

    public Builder wifiScans(final int wifiScans) {
      this.wifiScans = wifiScans;
      return this;
    }

    public Builder bleDevices(final int bleDevices) {
      this.bleDevices = bleDevices;
      return this;
    }

    public Builder lastSingleClick(final int lastSingleClick) {
      this.lastSingleClick = lastSingleClick;
      return this;
    }

    public Builder singleClickCount(final int singleClickCount) {
      this.singleClickCount = singleClickCount;
      return this;
    }

    public Builder dataLoggerStatus(final DeviceDataLoggerStatus dataLoggerStatus) {
      if (dataLoggerStatus != null) {
        this.dataLoggerStatus = dataLoggerStatus;
      }
      return this;
    }

    public Builder humidity(final int humidity) {
      this.humidity = humidity;
      return this;
    }

    public Builder bleChannel(final int bleChannel) {
      this.bleChannel = bleChannel;
      return this;
    }

    public KontaktTelemetry build() {
      return new KontaktTelemetry(this);
    }
  }
}
