package com.kontakt.sdk.android.ble.security.parser;

import com.kontakt.sdk.android.ble.security.CRCModbus;
import com.kontakt.sdk.android.ble.security.Flag;
import com.kontakt.sdk.android.ble.security.Operation;
import com.kontakt.sdk.android.ble.security.PayloadEncrypter;
import com.kontakt.sdk.android.ble.security.ResponseCode;
import com.kontakt.sdk.android.ble.security.exception.InvalidCRCException;
import com.kontakt.sdk.android.ble.security.exception.InvalidConfigException;
import com.kontakt.sdk.android.ble.security.exception.InvalidFlagException;
import java.nio.ByteBuffer;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNullOrEmpty;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

public class SimpleResponseParser {
  private static final int TOKEN_LENGTH = 4;
  private static final int RESULT_LENGTH = 1;
  private static final int IV_LENGTH = 16;
  private static final int CRC_LENGTH = 2;
  protected byte[] data;
  protected ResponseCode result;
  protected int token;
  protected Flag flag;
  protected Operation operation;
  private final String password;

  public static SimpleResponseParser of(byte[] response) throws InvalidConfigException {
    return new SimpleResponseParser(response, null);
  }

  public static SimpleResponseParser of(byte[] response, String password) throws InvalidConfigException {
    return new SimpleResponseParser(response, password);
  }

  protected SimpleResponseParser(byte[] response, String password) throws InvalidConfigException {
    this.password = password;
    extract(response);
  }

  private void extract(byte[] response) throws InvalidConfigException {
    ByteBuffer buffer = ByteBuffer.wrap(response);
    checkState(buffer.get() == 0x00, "Invalid protocol revision");
    flag = Flag.of(buffer.get());
    operation = Operation.of(buffer.get());
    switch (flag) {
      case ENCRYPTED:
        extractEncrypted(buffer);
        break;
      case NONE:
        extractPlaintext(buffer);
        break;
      default:
        throw new InvalidFlagException(flag, Flag.ENCRYPTED, Flag.NONE);
    }
  }

  private void extractPlaintext(ByteBuffer buffer) {
    short payloadLength = Short.reverseBytes(buffer.getShort());
    token = buffer.getInt();
    result = ResponseCode.of(buffer.get());
    int dataLength = payloadLength - TOKEN_LENGTH - RESULT_LENGTH; //without token and result
    data = new byte[dataLength];
    buffer.get(data);
  }

  private void extractEncrypted(ByteBuffer buffer) throws InvalidCRCException {
    checkNotNullOrEmpty(password, "password is null");
    short payloadLength = Short.reverseBytes(buffer.getShort());
    token = buffer.getInt();
    byte[] iv = ByteBuffer.allocate(16).putInt(token) //4 bytes
        .putInt(buffer.getInt()) //+ 4 bytes
        .putLong(buffer.getLong()) //+ 8 bytes = 16 bytes
        .array();
    int dataLength = payloadLength - IV_LENGTH - CRC_LENGTH; //without IV and CRC
    byte[] encryptedPayload = new byte[dataLength];
    buffer.get(encryptedPayload);
    byte[] decoded = new PayloadEncrypter(password, iv).decrypt(encryptedPayload);
    byte[] crc = new byte[2];
    buffer.get(crc);
    CRCModbus.assertCorrect(decoded, crc);
    ByteBuffer decodedBuffer = ByteBuffer.wrap(decoded);
    result = ResponseCode.of(decodedBuffer.get());
    data = new byte[decodedBuffer.remaining()];
    decodedBuffer.get(data);
  }

  public ResponseCode getResult() {
    return result;
  }

  public int getToken() {
    return Integer.reverseBytes(token);
  }

  public Operation getOperation() {
    return operation;
  }
}
