package com.kontakt.sdk.android.ble.filter.ibeacon;

import java.util.UUID;

/**
 * Filters are designed to enable more detailed devices search.
 *
 * There are several predefined filters including:
 * <ul>
 * <li>Proximity UUID filter - accepts devices only with desired Proximity UUID</li>
 * <li>Beacon Unique ID filter - every kontakt Beacon has assigned Unique Id to itself.
 * What is more this Unique Id may be read in Service Data in the Advertising Package.
 * Thus, filtering by this property is enabled as well. Please note that this filter
 * works well only for Beacons produces by Kontakt.</li>
 * <li>Device Name filter - accepts only devices named as specified during filter
 * creation and inclusion to ProximityManager configuration</li>
 * <li>Firmware filter - accepts only devices with specified version of firmware version.
 * Please note that this filter works well only with Kontakt beacons</li>
 * <li>Major filter - accepts only devices with specified major value</li>
 * <li>Minor filter - accepts only devices with specified minor value</li>
 * <li>Multi filter - accepts only devices with specified batch of parameters including:
 * device name, proximity UUID, MAC address, unique Id, firmware</li>
 * <li>Custom filter - this filter is introduced to perform custom filtering. Despite
 * the fact that other filters should meet your needs, you have possibility to create
 * your own filter.</li>
 * </ul>
 *
 * In Android version lower than {@link android.os.Build.VERSION_CODES#LOLLIPOP} filters
 * are included the form as they are specified. For the latest one and further the filters
 * are converted to {@link android.bluetooth.le.ScanFilter}s and passed to handle by the framework.
 */
public final class IBeaconFilters {
  private IBeaconFilters() {
  }

  /**
   * Creates new proximity UUID filter.
   *
   * @param proximityUUID the proximity uUID
   * @return the proximity UUID filter
   */
  public static ProximityUUIDFilter newProximityUUIDFilter(final UUID proximityUUID) {
    return new ProximityUUIDFilter(proximityUUID);
  }

  /**
   * Creates new beacon unique Id filter.
   *
   * @param beaconUniqueId the beacon unique id
   * @return the beacon unique id filter
   */
  public static IBeaconUniqueIdFilter newUniqueIdFilter(final String beaconUniqueId) {
    return new IBeaconUniqueIdFilter(beaconUniqueId);
  }

  /**
   * Creates new device name filter.
   *
   * @param beaconName the beacon name
   * @return the device name filter
   */
  public static DeviceNameFilter newDeviceNameFilter(final String beaconName) {
    return new DeviceNameFilter(beaconName);
  }

  /**
   * Creates new firmware filter.
   *
   * @param firmwareVersion the firmware version
   * @return the firmware filter
   */
  public static FirmwareFilter newFirmwareFilter(final String firmwareVersion) {
    return new FirmwareFilter(firmwareVersion);
  }

  /**
   * Creates new Major filter.
   *
   * @param major the major
   * @return the major filter
   */
  public static MajorFilter newMajorFilter(final int major) {
    return new MajorFilter(major);
  }

  /**
   * Creates new Minor filter.
   *
   * @param minor the minor
   * @return the minor filter
   */
  public static MinorFilter newMinorFilter(final int minor) {
    return new MinorFilter(minor);
  }

  /**
   * Creates new Multi Filter builder.
   *
   * @return the multi filter builder
   */
  public static IBeaconMultiFilter.Builder newMultiFilterBuilder() {
    return new IBeaconMultiFilter.Builder();
  }
}
