package com.kontakt.sdk.android.ble.discovery.secure_profile;

import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.FramePayload;
import com.kontakt.sdk.android.ble.manager.listeners.SecureProfileListener;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Provides abstraction for every Secure Profile payload resolver used in Kontakt.io SDK.
 */
public abstract class PayloadResolver {

  protected final byte payloadId;

  private ResolveListener listener;

  /**
   * Constructs new payload resolver with assigned payload identifier.
   *
   * @param payloadId the payload identifier.
   */
  public PayloadResolver(final byte payloadId) {
    this.payloadId = payloadId;
  }

  void attachListener(final ResolveListener listener) {
    this.listener = checkNotNull(listener, "Listener cannot be null");
  }

  /**
   * Resolves input payload by transforming it to {@link SecureProfile} object.
   * It is necessary to call {@link #onPayloadResolved(SecureProfile)} once payload is resolved.
   * Otherwise it won't be possible to get callbacks from {@link SecureProfileListener}.
   *
   * @param payload the input payload.
   */
  public abstract void resolve(FramePayload payload);

  /**
   * Call that method when payload is parsed successfully.
   *
   * @param profile parsed secure profile.
   */
  protected void onPayloadResolved(final SecureProfile profile) {
    if (listener != null) {
      listener.onResolved(profile);
    }
  }

  protected byte[] getServiceData(final FramePayload payload) {
    return payload.getData(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
  }

  protected String getName(final FramePayload payload) {
    final byte[] nameData = payload.getData(FrameDataType.LOCAL_NAME);
    if (nameData == null) {
      return null;
    }
    return new String(nameData);
  }

}
