package com.kontakt.sdk.android.ble.discovery.secure_profile;

import com.kontakt.sdk.android.ble.discovery.PropertyResolver;
import com.kontakt.sdk.android.ble.spec.Acceleration;
import com.kontakt.sdk.android.ble.spec.DeviceDataLoggerStatus;
import com.kontakt.sdk.android.ble.spec.KontaktTelemetry;
import com.kontakt.sdk.android.ble.spec.TelemetryError;

import static com.kontakt.sdk.android.common.util.ConversionUtils.asInt;
import static com.kontakt.sdk.android.common.util.ConversionUtils.asIntFromLittleEndianBytes;
import static com.kontakt.sdk.android.common.util.ConversionUtils.extractSubdata;

final class KontaktTLMResolver implements PropertyResolver<KontaktTelemetry> {

  private static final byte BASIC_SYSTEM_HEALTH_IDENTIFIER = 0x01;
  private static final byte ACCELEROMETER_IDENTIFIER = 0x02;
  private static final byte SCANNING_IDENTIFIER = 0x03;
  private static final byte MORE_SYSTEM_HEALTH_IDENTIFIER = 0x04;
  private static final byte SENSORS_IDENTIFIER = 0x05;
  private static final byte LIGHT_LEVEL_IDENTIFIER = 0x0A;
  private static final byte TEMPERATURE_IDENTIFIER = 0x0B;
  private static final byte BUTTON_CLICK_IDENTIFIER = 0x0D;
  private static final byte IDENTIFIED_BUTTON_CLICK_IDENTIFIER = 0x11;
  private static final byte DATA_LOGGER_STATUS_IDENTIFIER = 0x10;
  private static final byte HUMIDITY_IDENTIFIER = 0x12;
  private static final byte TEMPERATURE_16_BIT_IDENTIFIER = 0x13;
  private static final byte BLE_CHANNEL_IDENTIFIER = 0x14;

  private static final int BASIC_SYSTEM_HEALTH_FIELD_LENGTH = 6;
  private static final int ACCELEROMETER_FIELD_LENGTH = 9;
  private static final int SCANNING_FIELD_LENGTH = 5;
  private static final int MORE_SYSTEM_HEALTH_FIELD_LENGTH = 10;
  private static final int SENSORS_FIELD_LENGTH = 3;
  private static final int LIGHT_LEVEL_FIELD_LENGTH = 2;
  private static final int TEMPERATURE_FIELD_LENGTH = 2;
  private static final int BUTTON_CLICK_FIELD_LENGTH = 3;
  private static final int IDENTIFIED_BUTTON_CLICK_FIELD_LENGTH = 4;
  private static final int DATA_LOGGER_STATUS_FIELD_LENGTH = 2;
  private static final int HUMIDITY_FIELD_LENGTH = 2;
  private static final int TEMPERATURE_16_BIT_FIELD_LENGTH = 3;
  private static final int BLE_CHANNEL_FIELD_LENGTH = 2;

  private static final int FIELDS_PAYLOAD_OFFSET = 3;

  @Override
  public KontaktTelemetry parse(byte[] packet) {
    if (packet == null || packet.length == 0) {
      return null;
    }

    final KontaktTelemetry.Builder builder = new KontaktTelemetry.Builder();

    byte[] fieldsData = extractSubdata(packet, FIELDS_PAYLOAD_OFFSET, packet.length - FIELDS_PAYLOAD_OFFSET);
    while (fieldsData != null && fieldsData.length > 0) {
      final int fieldLength = asInt(fieldsData[0]);
      final byte[] field = extractSubdata(fieldsData, 1, fieldLength);
      if (field == null) break;
      final byte identifier = field[0];

      // Basic System Health
      if (identifier == BASIC_SYSTEM_HEALTH_IDENTIFIER && field.length >= BASIC_SYSTEM_HEALTH_FIELD_LENGTH) {
        resolveBasicSystemHealth(builder, field);
      }

      // Accelerometer data
      if (identifier == ACCELEROMETER_IDENTIFIER && field.length >= ACCELEROMETER_FIELD_LENGTH) {
        resolveAccelerometerData(builder, field);
      }

      // Scanning data
      if (identifier == SCANNING_IDENTIFIER && field.length >= SCANNING_FIELD_LENGTH) {
        resolveScanningData(builder, field);
      }

      // More System Health
      if (identifier == MORE_SYSTEM_HEALTH_IDENTIFIER && field.length >= MORE_SYSTEM_HEALTH_FIELD_LENGTH) {
        resolveMoreSystemHealth(builder, field);
      }

      // Sensors
      if (identifier == SENSORS_IDENTIFIER && field.length >= SENSORS_FIELD_LENGTH) {
        resolveSensors(builder, field);
      }

      // Light Level
      if (identifier == LIGHT_LEVEL_IDENTIFIER && field.length >= LIGHT_LEVEL_FIELD_LENGTH) {
        resolveLightLevel(builder, field);
      }

      // Temperature
      if (identifier == TEMPERATURE_IDENTIFIER && field.length >= TEMPERATURE_FIELD_LENGTH) {
        resolveTemperature(builder, field);
      }

      // Button Click
      if (identifier == BUTTON_CLICK_IDENTIFIER && field.length >= BUTTON_CLICK_FIELD_LENGTH) {
        resolveButtonClick(builder, field);
      }

      // Identified Button Click
      if (identifier == IDENTIFIED_BUTTON_CLICK_IDENTIFIER && field.length >= IDENTIFIED_BUTTON_CLICK_FIELD_LENGTH) {
        resolveIdentifiedButtonClick(builder, field);
      }

      // Data Logging
      if (identifier == DATA_LOGGER_STATUS_IDENTIFIER && field.length >= DATA_LOGGER_STATUS_FIELD_LENGTH) {
        resolveDataLoggerStatus(builder, field);
      }

      // Humidity
      if (identifier == HUMIDITY_IDENTIFIER && field.length >= HUMIDITY_FIELD_LENGTH) {
        resolveHumidity(builder, field);
      }

      // Temperature 16 Bit
      if (identifier == TEMPERATURE_16_BIT_IDENTIFIER && field.length >= TEMPERATURE_16_BIT_FIELD_LENGTH) {
        resolveTemperature16Bit(builder, field);
      }

      // BLE Channel
      if (identifier == BLE_CHANNEL_IDENTIFIER && field.length >= BLE_CHANNEL_FIELD_LENGTH) {
        resolveBleChannel(builder, field);
      }

      // Extract remaining fields data
      fieldsData = extractSubdata(fieldsData, field.length + 1, fieldsData.length - (field.length + 1));
    }

    return builder.build();
  }

  private void resolveBasicSystemHealth(KontaktTelemetry.Builder builder, byte[] field) {
    // Timestamp
    final int timestamp = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
    builder.timestamp(timestamp);

    // Battery level
    final int batteryLevel = asInt(field[5]);
    builder.batteryLevel(batteryLevel);
  }

  private void resolveAccelerometerData(KontaktTelemetry.Builder builder, byte[] field) {
    // Sensitivity
    final int sensitivity = asInt(field[1]);
    builder.sensitivity(sensitivity);

    // Acceleration
    final Acceleration acceleration = new Acceleration(extractSubdata(field, 2, 3));
    builder.acceleration(acceleration);

    // Last Double Tap
    final int lastDoubleTap = asIntFromLittleEndianBytes(extractSubdata(field, 5, 2));
    builder.lastDoubleTap(lastDoubleTap);

    // Last Threshold
    final int lastThreshold = asIntFromLittleEndianBytes(extractSubdata(field, 7, 2));
    builder.lastThreshold(lastThreshold);
  }

  private void resolveScanningData(KontaktTelemetry.Builder builder, byte[] field) {
    // BLE scans
    final int bleScans = asInt(field[1]);
    builder.bleScans(bleScans);

    // WiFi scans
    final int wifiScans = asInt(field[2]);
    builder.wifiScans(wifiScans);

    // BLE Devices
    final int bleDevices = asIntFromLittleEndianBytes(extractSubdata(field, 3, 2));
    builder.bleDevices(bleDevices);
  }

  private void resolveMoreSystemHealth(KontaktTelemetry.Builder builder, byte[] field) {
    // Timestamp
    final int timestamp = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
    builder.timestamp(timestamp);

    // Uptime
    final int uptime = asIntFromLittleEndianBytes(extractSubdata(field, 5, 2));
    builder.uptime(uptime);

    // System Load
    final int systemLoad = asInt(field[7]);
    builder.systemLoad(systemLoad);

    // Error
    final TelemetryError error = TelemetryError.fromValue(asInt(extractSubdata(field, 8, 2)));
    builder.error(error);
  }

  private void resolveSensors(KontaktTelemetry.Builder builder, byte[] field) {
    // Light Sensor
    final int lightSensor = asInt(field[1]);
    builder.lightSensor(lightSensor);

    // Temperature
    final int temperature = field[2];
    builder.temperature(temperature);
  }

  private void resolveLightLevel(KontaktTelemetry.Builder builder, byte[] field) {
    // Light Level
    final int light = asInt(field[1]);
    builder.lightSensor(light);
  }

  private void resolveTemperature(KontaktTelemetry.Builder builder, byte[] field) {
    // Temperature
    final int temperature = asInt(field[1]);
    builder.temperature(temperature);
  }

  private void resolveButtonClick(KontaktTelemetry.Builder builder, byte[] field) {
    // Click Time
    final int singleClick = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
    builder.lastSingleClick(singleClick < 0xFFFF ? singleClick : 0);
  }

  private void resolveIdentifiedButtonClick(KontaktTelemetry.Builder builder, byte[] field) {
    // Click Count
    final int clickCount = asInt(field[1]);
    builder.singleClickCount(clickCount);

    // Click Time
    final int singleClick = asIntFromLittleEndianBytes(extractSubdata(field, 2, 2));
    builder.lastSingleClick(singleClick < 0xFFFF ? singleClick : 0);
  }

  private void resolveDataLoggerStatus(KontaktTelemetry.Builder builder, byte[] field) {
    // Data Logging Status
    final int loggingStatus = asInt(field[1]);
    builder.dataLoggerStatus(loggingStatus == 1 ? DeviceDataLoggerStatus.ENABLED : DeviceDataLoggerStatus.DISABLED);
  }

  private void resolveHumidity(KontaktTelemetry.Builder builder, byte[] field) {
    // Humidity percentage value
    final int humidity = asInt(field[1]);
    builder.humidity(humidity);
  }

  private void resolveTemperature16Bit(KontaktTelemetry.Builder builder, byte[] field) {
    // Temperature 16 Bit
    final int temperature = asIntFromLittleEndianBytes(extractSubdata(field, 1, 2));
    builder.temperature(temperature / 256);
  }

  private void resolveBleChannel(KontaktTelemetry.Builder builder, byte[] field) {
    // BLE Channel value
    final int bleChannel = asInt(field[1]);
    builder.bleChannel(bleChannel);
  }

}
