package com.kontakt.sdk.android.ble.dfu;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Helper class containing single DFU bytes transaction
 */
class Transaction {

  static final int DEFAULT_SIZE = 2048;
  static final int BYTES_CHUNK_LENGTH = 20;

  private final byte[] fileBytes;
  private final int size;
  private final int storedBytesCount;

  public Transaction(byte[] fileBytes, int storedBytesCount) {
    this.fileBytes = fileBytes;
    this.storedBytesCount = storedBytesCount;
    this.size = calculateTransactionLength(fileBytes, storedBytesCount);
  }

  /**
   * @return Calculated size for this transaction.
   */
  public int getSize() {
    return size;
  }

  /**
   * @return List of chunks to send as part of this transaction. Single chunk contains {@link #BYTES_CHUNK_LENGTH} bytes.
   */
  public List<byte[]> getChunksToSend() {
    List<byte[]> chunksToSend = new ArrayList<>();

    int index = storedBytesCount;
    int count = (int) Math.floor((double) size / (double) BYTES_CHUNK_LENGTH);
    int rest = size % BYTES_CHUNK_LENGTH;
    for (int i = 0; i < count; i++) {
      chunksToSend.add(Arrays.copyOfRange(fileBytes, index, index + BYTES_CHUNK_LENGTH));
      index += BYTES_CHUNK_LENGTH;
    }

    if (rest != 0) {
      index = storedBytesCount + (count * BYTES_CHUNK_LENGTH);
      chunksToSend.add(Arrays.copyOfRange(fileBytes, index, index + rest));
    }

    return chunksToSend;
  }

  private int calculateTransactionLength(byte[] fileBytes, int storedSize) {
    if (storedSize == 0) {
      return DEFAULT_SIZE;
    }
    int length = fileBytes.length - storedSize;
    return length >= DEFAULT_SIZE ? DEFAULT_SIZE : length;
  }
}
