package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.model.SecureProfileFutureUID;
import com.kontakt.sdk.android.common.model.SecureProfileUid;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

class SecureProfileCacheResolveStrategy implements SecureProfileResolveStrategy {

  private final FutureShufflesCache cache;

  SecureProfileCacheResolveStrategy(FutureShufflesCache cache) {
    this.cache = cache;
  }

  @Override
  public List<SecureProfileFutureUID> resolve(Map<SecureProfileUid, SecureProfileResolveRequest> requests) throws Exception {
    List<SecureProfileUid> toResolve = profilesToResolve(requests);
    if (toResolve.isEmpty()) {
      return Collections.emptyList();
    }

    List<SecureProfileFutureUID> futureIds = resolveProfiles(toResolve);
    updateRequests(requests, futureIds);
    return futureIds;
  }

  private List<SecureProfileUid> profilesToResolve(final Map<SecureProfileUid, SecureProfileResolveRequest> requests) {
    List<SecureProfileUid> toResolve = new ArrayList<>();
    for (Map.Entry<SecureProfileUid, SecureProfileResolveRequest> entry : requests.entrySet()) {
      SecureProfileResolveRequest request = entry.getValue();
      if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
        continue;
      }
      toResolve.add(entry.getKey());
    }
    return toResolve;
  }

  private List<SecureProfileFutureUID> resolveProfiles(List<SecureProfileUid> uids) {
    List<SecureProfileFutureUID> futureIds = new ArrayList<>();
    for (SecureProfileUid secureUID : uids) {
      ResolvedId resolvedId = cache.get(secureUID.toString());
      if (resolvedId == null || FutureShufflesCache.PHANTOM_ENTRY.equals(resolvedId)) {
        continue;
      }

      SecureProfileFutureUID futureUID = new SecureProfileFutureUID.Builder()
          .futureIds(new ArrayList<SecureProfileUid>())
          .uniqueId(resolvedId.getUniqueId())
          .queriedBy(secureUID)
          .resolved(resolvedId.getSecureProfileUID())
          .build();
      futureIds.add(futureUID);
    }
    return futureIds;
  }

  private void updateRequests(Map<SecureProfileUid, SecureProfileResolveRequest> requests, List<SecureProfileFutureUID> futureIds) {
    for (SecureProfileFutureUID id : futureIds) {
      updateRequest(requests, id);
    }
  }

  private void updateRequest(Map<SecureProfileUid, SecureProfileResolveRequest> requests, SecureProfileFutureUID id) {
    SecureProfileResolveRequest request = requests.get(id.getQueriedBy());
    request.setResolvedBy(ResolverType.CACHE);
    if (ResolveRequestStatus.IGNORED == request.getStatus()) {
      return;
    }
    request.setStatus(ResolveRequestStatus.RESOLVED);
  }
}
