package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * This class represents an iBeacon-ID composed of 16-byte proximity UUID, 2-byte major and 2-byte minor.
 */
public class IBeaconId implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<IBeaconId> CREATOR = new Creator<IBeaconId>() {
    @Override
    public IBeaconId createFromParcel(Parcel source) {
      return new IBeaconId(source);
    }

    @Override
    public IBeaconId[] newArray(int size) {
      return new IBeaconId[size];
    }
  };

  private UUID proximity;
  private int major;
  private int minor;

  /**
   * Creates new iBeacon-ID object with specified proximity, major and minor.
   *
   * @param proximity the proximity UUID.
   * @param major the major value.
   * @param minor the minor value.
   * @return the iBeacon-ID instance.
   */
  public static IBeaconId of(UUID proximity, int major, int minor) {
    return new IBeaconId(proximity, major, minor);
  }

  /**
   * Creates new iBeacon-ID object from {@link IBeaconDevice} instance.
   *
   * @param iBeaconDevice the iBeacon device instance.
   * @return the iBeacon-ID instance.
   */
  public static IBeaconId fromDevice(IBeaconDevice iBeaconDevice) {
    UUID proximity = iBeaconDevice.getProximityUUID();
    int major = iBeaconDevice.getMajor();
    int minor = iBeaconDevice.getMinor();
    return new IBeaconId(proximity, major, minor);
  }

  /**
   * Creates new iBeacon-ID object from String value.
   *
   * @param queriedBy the string value in proximity:major:minor format.
   * @return the iBeacon-ID instance.
   * @throws NullPointerException if input String is null.
   * @throws IllegalArgumentException if input String has invalid format.
   */
  public static IBeaconId fromQueriedBy(String queriedBy) {
    SDKPreconditions.checkNotNull(queriedBy, "queriedBy is null");
    String[] split = queriedBy.split(":");
    if (split.length != 3) {
      throw new IllegalArgumentException("queriedBy is not valid format");
    }
    UUID proximity = UUID.fromString(split[0]);
    int major = Integer.parseInt(split[1]);
    int minor = Integer.parseInt(split[2]);
    return new IBeaconId(proximity, major, minor);
  }

  private IBeaconId(UUID proximity, int major, int minor) {
    this.proximity = proximity;
    this.major = major;
    this.minor = minor;
  }

  protected IBeaconId(Parcel in) {
    this.proximity = (UUID) in.readSerializable();
    this.major = in.readInt();
    this.minor = in.readInt();
  }

  public UUID getProximity() {
    return proximity;
  }

  public int getMajor() {
    return major;
  }

  public int getMinor() {
    return minor;
  }

  // do not change implementation of toString() method, it is used as http request param as well
  @Override
  public String toString() {
    return proximity.toString() + ":" + major + ":" + minor;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof IBeaconId)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    IBeaconId beaconId = (IBeaconId) o;
    return SDKEqualsBuilder.start().equals(proximity, beaconId.proximity).equals(major, beaconId.major).equals(minor, beaconId.minor).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(proximity).append(major).append(minor).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.proximity);
    dest.writeInt(this.major);
    dest.writeInt(this.minor);
  }
}
