package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;

import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * Model class that represents JSON structure of Kontakt.io device.
 * <br><br>
 * Device is a property of {@link Manager}. It acts also as a container for {@link Action} objects
 * and may be assigned to certain {@link Venue}. We can obtain devices via {@link KontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link Device.Builder}.
 */
public class Device implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Device> CREATOR = new Creator<Device>() {
    @Override
    public Device createFromParcel(Parcel source) {
      return new Device(source);
    }

    @Override
    public Device[] newArray(int size) {
      return new Device[size];
    }
  };

  private String uniqueId;
  private UUID id;
  @SerializedName("futureId") private Shuffles shuffles;
  private String mac;
  private UUID secureProximity;
  private String secureNamespace;
  private Config config;
  private String firmware;
  private String alias;
  private List<String> tags;
  private DeviceType deviceType;
  private Specification specification;
  private Model model;
  private String product;
  private UUID managerId;
  private Venue venue;
  private int actionsCount;
  private Access access;
  private String lat;
  private String lng;
  private String queriedBy;
  private List<SubscriptionPlan> subscriptionPlans;
  private Map<String, String> metadata;
  private String orderId;

  public static Builder builder() {
    return new Builder();
  }

  private Device() {
    this(new Builder());
  }

  Device(Builder builder) {
    this.uniqueId = builder.uniqueId;
    this.id = builder.id;
    this.shuffles = builder.shuffles;
    this.mac = builder.mac;
    this.secureProximity = builder.secureProximity;
    this.secureNamespace = builder.secureNamespace;
    this.config = builder.config;
    this.firmware = builder.firmware;
    this.alias = builder.alias;
    this.tags = builder.tags;
    this.deviceType = builder.deviceType;
    this.specification = builder.specification;
    this.model = builder.model;
    this.product = builder.product;
    this.managerId = builder.managerId;
    this.venue = builder.venue;
    this.actionsCount = builder.actionsCount;
    this.access = builder.access;
    this.lat = builder.lat;
    this.lng = builder.lng;
    this.queriedBy = builder.queriedBy;
    this.subscriptionPlans = builder.subscriptionPlans;
    this.metadata = builder.metadata;
    this.orderId = builder.orderId;
  }

  protected Device(Parcel in) {
    this.uniqueId = in.readString();
    this.id = (UUID) in.readSerializable();
    this.shuffles = in.readParcelable(Shuffles.class.getClassLoader());
    this.mac = in.readString();
    this.secureProximity = (UUID) in.readSerializable();
    this.secureNamespace = in.readString();
    this.config = in.readParcelable(Config.class.getClassLoader());
    this.firmware = in.readString();
    this.alias = in.readString();
    this.tags = new ArrayList<>();
    in.readList(this.tags, String.class.getClassLoader());
    int tmpDeviceType = in.readInt();
    this.deviceType = tmpDeviceType == -1 ? null : DeviceType.values()[tmpDeviceType];
    int tmpSpecification = in.readInt();
    this.specification = tmpSpecification == -1 ? null : Specification.values()[tmpSpecification];
    int tmpModel = in.readInt();
    this.model = tmpModel == -1 ? null : Model.values()[tmpModel];
    this.product = in.readString();
    this.managerId = (UUID) in.readSerializable();
    this.venue = in.readParcelable(Venue.class.getClassLoader());
    this.actionsCount = in.readInt();
    int tmpAccess = in.readInt();
    this.access = tmpAccess == -1 ? null : Access.values()[tmpAccess];
    this.lat = in.readString();
    this.lng = in.readString();
    this.queriedBy = in.readString();
    this.subscriptionPlans = new ArrayList<>();
    in.readList(this.subscriptionPlans, SubscriptionPlan.class.getClassLoader());

    int metadataSize = in.readInt();
    if (metadataSize != 0) {
      this.metadata = new HashMap<>(metadataSize);
      for (int i = 0; i < metadataSize; i++) {
        String key = in.readString();
        String value = in.readString();
        this.metadata.put(key, value);
      }
    }

    this.orderId = in.readString();
  }

  @Override
  public boolean equals(Object o) {
    if (o == this) {
      return true;
    }
    if (o == null || !(o instanceof Device)) {
      return false;
    }
    Device device = (Device) o;

    return SDKEqualsBuilder.start()
        .equals(uniqueId, device.uniqueId)
        .equals(firmware, device.firmware)
        .equals(alias, device.alias)
        .equals(lat, device.lat)
        .equals(lng, device.lng)
        .equals(queriedBy, device.queriedBy)
        .equals(access, device.access)
        .equals(venue, device.venue)
        .equals(actionsCount, device.actionsCount)
        .equals(id, device.id)
        .equals(deviceType, device.deviceType)
        .equals(managerId, device.managerId)
        .equals(specification, device.specification)
        .equals(model, device.model)
        .equals(product, device.product)
        .equals(mac, device.mac)
        .equals(secureProximity, device.secureProximity)
        .equals(secureNamespace, device.secureNamespace)
        .equals(shuffles, device.shuffles)
        .equals(config, device.config)
        .equals(subscriptionPlans, device.subscriptionPlans)
        .equals(orderId, device.orderId)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(uniqueId)
        .append(firmware)
        .append(alias)
        .append(lat)
        .append(lng)
        .append(queriedBy)
        .append(access)
        .append(venue)
        .append(actionsCount)
        .append(id)
        .append(deviceType)
        .append(managerId)
        .append(specification)
        .append(model)
        .append(product)
        .append(mac)
        .append(secureProximity)
        .append(secureNamespace)
        .append(shuffles)
        .append(config)
        .append(subscriptionPlans)
        .append(orderId)
        .build();
  }

  @Override
  public String toString() {
    return "Device{"
        + "uniqueId='" + uniqueId + '\''
        + ", id=" + id
        + ", shuffles=" + shuffles
        + ", mac='" + mac + '\''
        + ", secureProximity=" + secureProximity
        + ", secureNamespace='" + secureNamespace + '\''
        + ", config=" + config
        + ", firmware='" + firmware + '\''
        + ", alias='" + alias + '\''
        + ", tags=" + tags
        + ", deviceType=" + deviceType
        + ", specification=" + specification
        + ", model=" + model
        + ", product=" + product
        + ", managerId=" + managerId
        + ", venue=" + venue
        + ", actionsCount=" + actionsCount
        + ", access=" + access
        + ", lat='" + lat + '\''
        + ", lng='" + lng + '\''
        + ", queriedBy='" + queriedBy + '\''
        + ", subscriptionPlans=" + subscriptionPlans
        + ", metadata=" + metadata
        + ", orderId='" + orderId + '\''
        + '}';
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.uniqueId);
    dest.writeSerializable(this.id);
    dest.writeParcelable(this.shuffles, flags);
    dest.writeString(this.mac);
    dest.writeSerializable(this.secureProximity);
    dest.writeString(this.secureNamespace);
    dest.writeParcelable(this.config, flags);
    dest.writeString(this.firmware);
    dest.writeString(this.alias);
    dest.writeList(this.tags);
    dest.writeInt(this.deviceType == null ? -1 : this.deviceType.ordinal());
    dest.writeInt(this.specification == null ? -1 : this.specification.ordinal());
    dest.writeInt(this.model == null ? -1 : this.model.ordinal());
    dest.writeString(this.product);
    dest.writeSerializable(this.managerId);
    dest.writeParcelable(this.venue, flags);
    dest.writeInt(this.actionsCount);
    dest.writeInt(this.access == null ? -1 : this.access.ordinal());
    dest.writeString(this.lat);
    dest.writeString(this.lng);
    dest.writeString(this.queriedBy);
    dest.writeList(this.subscriptionPlans);

    int metadataSize = this.metadata != null ? this.metadata.size() : 0;
    dest.writeInt(metadataSize);
    if (metadataSize != 0) {
      for (Map.Entry<String, String> entry : this.metadata.entrySet()) {
        dest.writeString(entry.getKey());
        dest.writeString(entry.getValue());
      }
    }

    dest.writeString(this.orderId);
  }

  /**
   * Updates old configuration with the new one.
   *
   * @param config new configuration.
   */
  public void applyConfig(final Config config) {
    this.config.applyConfig(config);
  }

  /**
   * Updates old secure configuration with the new one.
   *
   * @param secureConfig new secure configuration.
   */
  public void applySecureConfig(final Config secureConfig) {
    config.applySecureConfig(secureConfig);
  }

  /**
   * Updates old credentials with the new one.
   *
   * @param credentials new credentials.
   */
  public void applyCredentials(final Credentials credentials) {
    SDKPreconditions.checkNotNull(credentials, "credentials cannot be null");
    config.changePassword(credentials.getPassword());
  }

  /**
   * Extracts {@link IBeaconFutureId} instance from {@link Shuffles} object, composed in this class.
   *
   * @return extracted IBeaconFutureId object.
   */
  public IBeaconFutureId extractIBeaconFutureId() {
    if (shuffles == null || !config.getProfiles().contains(DeviceProfile.IBEACON)) {
      return null;
    }

    final UUID proximity = config.getProximity();
    final int major = config.getMajor();
    final int minor = config.getMinor();
    final IBeaconId resolved = IBeaconId.of(proximity, major, minor);

    return new IBeaconFutureId.Builder().uniqueId(uniqueId)
        .queriedBy(queriedBy != null ? IBeaconId.fromQueriedBy(queriedBy) : null)
        .resolved(resolved)
        .futureIds(shuffles.getIBeaconShuffles())
        .build();
  }

  /**
   * Extracts {@link EddystoneFutureUID} instance from {@link Shuffles} object, composed in this class.
   *
   * @return extracted EddystoneFutureUID object.
   */
  public EddystoneFutureUID extractEddystoneFutureUID() {
    if (shuffles == null || !config.getProfiles().contains(DeviceProfile.EDDYSTONE)) {
      return null;
    }

    final String namespace = config.getNamespace();
    final String instanceId = config.getInstanceId();
    final EddystoneUid resolved = EddystoneUid.of(namespace, instanceId);

    return new EddystoneFutureUID.Builder().uniqueId(uniqueId)
        .queriedBy(queriedBy != null ? EddystoneUid.fromQueriedBy(queriedBy) : null)
        .resolved(resolved)
        .futureIds(shuffles.getEddystoneShuffles())
        .build();
  }

  /**
   * Extracts {@link EddystoneFutureUID} instance from {@link Shuffles} object, composed in this class.
   *
   * @return extracted EddystoneFutureUID object.
   */
  public SecureProfileFutureUID extractSecureProfileFutureUID() {
    if (shuffles == null || !config.getPackets().contains(PacketType.KONTAKT)) {
      return null;
    }

    final String namespace = config.getNamespace();
    final String instanceId = config.getInstanceId();
    final SecureProfileUid resolved = SecureProfileUid.of(namespace, instanceId);

    return new SecureProfileFutureUID.Builder()
        .uniqueId(uniqueId)
        .queriedBy(queriedBy != null ? SecureProfileUid.fromQueriedBy(queriedBy) : null)
        .resolved(resolved)
        .futureIds(shuffles.getSecureProfileShuffles())
        .build();
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public UUID getId() {
    return id;
  }

  public Shuffles getShuffles() {
    return shuffles;
  }

  public String getMac() {
    return mac;
  }

  public UUID getSecureProximity() {
    return secureProximity;
  }

  public String getSecureNamespace() {
    return secureNamespace;
  }

  public Config getConfig() {
    return config;
  }

  public String getFirmware() {
    return firmware;
  }

  public String getAlias() {
    return alias;
  }

  public List<String> getTags() {
    return tags;
  }

  public DeviceType getDeviceType() {
    return deviceType;
  }

  public Specification getSpecification() {
    return specification;
  }

  public Model getModel() {
    return model;
  }

  public String getProduct() {
    return product;
  }

  public UUID getManagerId() {
    return managerId;
  }

  public Venue getVenue() {
    return venue;
  }

  public int getActionsCount() {
    return actionsCount;
  }

  public Access getAccess() {
    return access;
  }

  public String getLatitude() {
    return lat;
  }

  public String getLongitude() {
    return lng;
  }

  public String getQueriedBy() {
    return queriedBy;
  }

  public List<SubscriptionPlan> getSubscriptionPlans() {
    return subscriptionPlans;
  }

  public Map<String, String> getMetadata() {
    return metadata;
  }

  public String getOrderId() {
    return orderId;
  }

  /**
   * Builder class that is used to build {@link Device} instances from values configured by the setters.
   */
  public static class Builder {
    String uniqueId;
    UUID id;
    Shuffles shuffles;
    String mac;
    UUID secureProximity;
    String secureNamespace;
    Config config;
    String firmware;
    String alias;
    List<String> tags;
    DeviceType deviceType;
    Specification specification;
    Model model;
    String product;
    UUID managerId;
    Venue venue;
    int actionsCount;
    Access access;
    String lat;
    String lng;
    String queriedBy;
    List<SubscriptionPlan> subscriptionPlans = new ArrayList<>();
    Map<String, String> metadata;
    String orderId;

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder shuffles(final Shuffles shuffles) {
      this.shuffles = shuffles;
      return this;
    }

    public Builder mac(final String mac) {
      this.mac = mac;
      return this;
    }

    public Builder secureProximity(final UUID secureProximity) {
      this.secureProximity = secureProximity;
      return this;
    }

    public Builder secureNamespace(final String secureNamespace) {
      this.secureNamespace = secureNamespace;
      return this;
    }

    public Builder config(final Config config) {
      this.config = SDKPreconditions.checkNotNull(config, "config cannot be null");
      return this;
    }

    public Builder firmware(final String firmware) {
      this.firmware = firmware;
      return this;
    }

    public Builder alias(final String alias) {
      this.alias = alias;
      return this;
    }

    public Builder tags(final List<String> tags) {
      this.tags = tags;
      return this;
    }

    public Builder deviceType(final DeviceType deviceType) {
      this.deviceType = deviceType;
      return this;
    }

    public Builder specification(final Specification specification) {
      this.specification = specification;
      return this;
    }

    public Builder model(final Model model) {
      this.model = model;
      return this;
    }

    public Builder product(final String product) {
      this.product = product;
      return this;
    }

    public Builder managerId(final UUID managerId) {
      this.managerId = managerId;
      return this;
    }

    public Builder venue(final Venue venue) {
      this.venue = venue;
      return this;
    }

    public Builder actionsCount(final int actionsCount) {
      SDKPreconditions.checkState(actionsCount >= 0, "actions count cannot be negative");
      this.actionsCount = actionsCount;
      return this;
    }

    public Builder access(final Access access) {
      this.access = access;
      return this;
    }

    public Builder latitude(final String lat) {
      this.lat = lat;
      return this;
    }

    public Builder longitude(final String lng) {
      this.lng = lng;
      return this;
    }

    public Builder queriedBy(final String queriedBy) {
      this.queriedBy = queriedBy;
      return this;
    }

    public Builder subscriptionPlans(final List<SubscriptionPlan> subscriptionPlans) {
      this.subscriptionPlans = subscriptionPlans;
      return this;
    }

    public Builder metadata(final Map<String, String> metadata) {
      this.metadata = metadata;
      return this;
    }

    public Builder orderId(final String orderId) {
      this.orderId = orderId;
      return this;
    }

    public Device build() {
      return new Device(this);
    }
  }
}
