package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.venues.CreateVenueRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.DeleteVenueRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.ShareVenueRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.UnshareVenueRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.UpdateVenueRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.VenueImageRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.VenueRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.venues.VenuesRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.common.model.Venue;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.UUID;

/**
 * An abstract representation of Venues API endpoint. Provides executors that can be
 * used to handle all venue-related requests.
 */
public class VenuesApi {

  private final VenuesService venuesService;

  /**
   * Constructs Venues API endpoint initialized with corresponding service class.
   *
   * @param venuesService the venues API service.
   */
  public VenuesApi(final VenuesService venuesService) {
    this.venuesService = venuesService;
  }

  /**
   * Provides executor that can be used to obtain single venue by ID.
   *
   * @param id the venue's unique identifier.
   * @return the request executor.
   */
  public VenueRequestExecutor fetch(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new VenueRequestExecutor(venuesService, id);
  }

  /**
   * Provides executor that can be used to obtain venues.
   *
   * @return the request executor.
   */
  public VenuesRequestExecutor fetch() {
    return new VenuesRequestExecutor(venuesService);
  }

  /**
   * Provides executor that can be used to load venue's image file.
   *
   * @param id the venue's unique identifier.
   * @return the request executor.
   */
  public VenueImageRequestExecutor imageOf(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new VenueImageRequestExecutor(venuesService, id);
  }

  /**
   * Provides executor that can be used to create new venue.
   *
   * @param venue the venue.
   * @return the request executor.
   */
  public CreateVenueRequestExecutor create(final Venue venue) {
    SDKPreconditions.checkNotNull(venue, "venue cannot be null");
    return new CreateVenueRequestExecutor(venuesService, venue);
  }

  /**
   * Provides executor that can be used to update a venue.
   *
   * @param id the venue's unique identifier.
   * @return the request executor.
   */
  public UpdateVenueRequestExecutor update(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new UpdateVenueRequestExecutor(venuesService, id);
  }

  /**
   * Provides executor that can be used to delete a venue.
   *
   * @param id the venue's unique identifier.
   * @return the request executor.
   */
  public DeleteVenueRequestExecutor delete(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new DeleteVenueRequestExecutor(venuesService, id);
  }

  /**
   * Provides executor that can be used to share venues with other managers.
   *
   * @param ids venue unique identifiers.
   * @return the request executor.
   */
  public ShareVenueRequestExecutor share(final UUID... ids) {
    SDKPreconditions.checkNotNull(ids, "IDs cannot be null");
    return new ShareVenueRequestExecutor(venuesService, ids);
  }

  /**
   * Provides executor that can be used to share venues with other managers.
   *
   * @param ids venue unique identifiers.
   * @return the request executor.
   */
  public ShareVenueRequestExecutor share(final List<UUID> ids) {
    SDKPreconditions.checkNotNull(ids, "IDs cannot be null");
    return new ShareVenueRequestExecutor(venuesService, ids);
  }

  /**
   * Provides executor that can be used to unshare venues from managers.
   *
   * @param ids venue unique identifiers.
   * @return the request executor.
   */
  public UnshareVenueRequestExecutor unshare(final UUID... ids) {
    SDKPreconditions.checkNotNull(ids, "IDs cannot be null");
    return new UnshareVenueRequestExecutor(venuesService, ids);
  }

  /**
   * Provides executor that can be used to unshare venues from managers.
   *
   * @param ids venue unique identifiers.
   * @return the request executor.
   */
  public UnshareVenueRequestExecutor unshare(final List<UUID> ids) {
    SDKPreconditions.checkNotNull(ids, "IDs cannot be null");
    return new UnshareVenueRequestExecutor(venuesService, ids);
  }
}
