package com.kontakt.sdk.android.ble.discovery.secure_profile;

import com.kontakt.sdk.android.ble.discovery.PropertyResolver;
import com.kontakt.sdk.android.ble.spec.Acceleration;
import com.kontakt.sdk.android.ble.spec.KontaktTelemetry;
import com.kontakt.sdk.android.ble.spec.TelemetryError;

import static com.kontakt.sdk.android.common.util.ConversionUtils.asInt;
import static com.kontakt.sdk.android.common.util.ConversionUtils.asIntFromLittleEndianBytes;
import static com.kontakt.sdk.android.common.util.ConversionUtils.extractSubdata;

final class KontaktTLMResolver implements PropertyResolver<KontaktTelemetry> {

  private static final byte BASIC_SYSTEM_HEALTH_IDENTIFIER = 0x01;
  private static final byte ACCELEROMETER_IDENTIFIER = 0x02;
  private static final byte SCANNING_IDENTIFIER = 0x03;
  private static final byte MORE_SYSTEM_HEALTH_IDENTIFIER = 0x04;
  private static final byte SENSORS_IDENTIFIER = 0x05;

  private static final int BASIC_SYSTEM_HEALTH_FIELD_LENGTH = 6;
  private static final int ACCELEROMETER_FIELD_LENGTH = 9;
  private static final int SCANNING_FIELD_LENGTH = 5;
  private static final int MORE_SYSTEM_HEALTH_FIELD_LENGTH = 10;
  private static final int SENSORS_FIELD_LENGTH = 3;

  private static final int FIELDS_PAYLOAD_OFFSET = 3;

  @Override
  public KontaktTelemetry parse(byte[] packet) {
    if (packet == null || packet.length == 0) {
      return null;
    }

    final KontaktTelemetry.Builder builder = new KontaktTelemetry.Builder();

    byte[] fieldsData = extractSubdata(packet, FIELDS_PAYLOAD_OFFSET, packet.length - FIELDS_PAYLOAD_OFFSET);
    while (fieldsData != null && fieldsData.length > 0) {
      final int fieldLength = asInt(fieldsData[0]);
      final byte[] field = extractSubdata(fieldsData, 1, fieldLength);
      if (field == null) break;
      final byte identifier = field[0];

      // Basic System Health
      if (identifier == BASIC_SYSTEM_HEALTH_IDENTIFIER && field.length >= BASIC_SYSTEM_HEALTH_FIELD_LENGTH) {
        resolveBasicSystemHealth(builder, field);
      }

      // Accelerometer data
      if (identifier == ACCELEROMETER_IDENTIFIER && field.length >= ACCELEROMETER_FIELD_LENGTH) {
        resolveAccelerometerData(builder, field);
      }

      // Scanning data
      if (identifier == SCANNING_IDENTIFIER && field.length >= SCANNING_FIELD_LENGTH) {
        resolveScanningData(builder, field);
      }

      // More System Health
      if (identifier == MORE_SYSTEM_HEALTH_IDENTIFIER && field.length >= MORE_SYSTEM_HEALTH_FIELD_LENGTH) {
        resolveMoreSystemHealth(builder, field);
      }

      // Sensors
      if (identifier == SENSORS_IDENTIFIER && field.length >= SENSORS_FIELD_LENGTH) {
        resolveSensors(builder, field);
      }

      // Extract remaining fields data
      fieldsData = extractSubdata(fieldsData, field.length + 1, fieldsData.length - (field.length + 1));
    }

    return builder.build();
  }

  private void resolveBasicSystemHealth(KontaktTelemetry.Builder builder, byte[] field) {
    // Timestamp
    final int timestamp = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
    builder.timestamp(timestamp);

    // Battery level
    final int batteryLevel = asInt(field[5]);
    builder.batteryLevel(batteryLevel);
  }

  private void resolveAccelerometerData(KontaktTelemetry.Builder builder, byte[] field) {
    // Sensitivity
    final int sensitivity = asInt(field[1]);
    builder.sensitivity(sensitivity);

    // Acceleration
    final Acceleration acceleration = new Acceleration(extractSubdata(field, 2, 3));
    builder.acceleration(acceleration);

    // Last Double Tap
    final int lastDoubleTap = asIntFromLittleEndianBytes(extractSubdata(field, 5, 2));
    builder.lastDoubleTap(lastDoubleTap);

    // Last Threshold
    final int lastThreshold = asIntFromLittleEndianBytes(extractSubdata(field, 7, 2));
    builder.lastThreshold(lastThreshold);
  }

  private void resolveScanningData(KontaktTelemetry.Builder builder, byte[] field) {
    // BLE scans
    final int bleScans = asInt(field[1]);
    builder.bleScans(bleScans);

    // WiFi scans
    final int wifiScans = asInt(field[2]);
    builder.wifiScans(wifiScans);

    // BLE Devices
    final int bleDevices = asIntFromLittleEndianBytes(extractSubdata(field, 3, 2));
    builder.bleDevices(bleDevices);
  }

  private void resolveMoreSystemHealth(KontaktTelemetry.Builder builder, byte[] field) {
    // Timestamp
    final int timestamp = asIntFromLittleEndianBytes(extractSubdata(field, 1, 4));
    builder.timestamp(timestamp);

    // Uptime
    final int uptime = asIntFromLittleEndianBytes(extractSubdata(field, 5, 2));
    builder.uptime(uptime);

    // System Load
    final int systemLoad = asInt(field[7]);
    builder.systemLoad(systemLoad);

    // Error
    final TelemetryError error = TelemetryError.fromValue(asInt(extractSubdata(field, 8, 2)));
    builder.error(error);
  }

  private void resolveSensors(KontaktTelemetry.Builder builder, byte[] field) {
    // Light Sensor
    final int lightSensor = asInt(field[1]);
    builder.lightSensor(lightSensor);

    // Temperature
    final int temperature = field[2];
    builder.temperature(temperature);
  }

}
