package com.kontakt.sdk.android.ble.discovery;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.ISecureProfile;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class ShuffledSecureProfileResolver {

  public interface ResolveCallback {
    void onResolved(ISecureProfile profile);
  }

  private final FutureShufflesCache cache;
  private final ResolveCallback callback;

  public ShuffledSecureProfileResolver(ResolveCallback resolveCallback, FutureShufflesCache cache) {
    this.callback = checkNotNull(resolveCallback);
    this.cache = checkNotNull(cache);
    this.cache.addCallback(resolveCallback);
  }

  public void resolve(ISecureProfile originalDevice) {
    if (originalDevice == null) {
      return;
    }
    if (!originalDevice.isShuffled()) {
      callback.onResolved(originalDevice);
      return;
    }

    ResolvedId resolvedId = cache.get(originalDevice);
    if (resolvedId == null) {
      cache.addResolveRequest(originalDevice);
      return;
    }
    if (FutureShufflesCache.PHANTOM_ENTRY.equals(resolvedId)) {
      return;
    }

    if (originalDevice.getUniqueId() == null) {
      ISecureProfile resolvedDevice = createUpdatedDevice(originalDevice, resolvedId);
      callback.onResolved(resolvedDevice);
    } else {
      // if origin.uniqueId != null && origin.isShuffled
      // then the event was sent from resolvers
      // and can be directly further processed
      callback.onResolved(originalDevice);
    }
  }

  public void onDeviceLost(ISecureProfile secureProfile) {
    ResolvedId resolvedId = cache.get(secureProfile);
    // not resolved yet
    if (resolvedId == null) {
      cache.markIgnored(secureProfile);
    }
  }

  public void disable() {
    cache.finishResolveRunners();
  }

  private ISecureProfile createUpdatedDevice(ISecureProfile secureProfile, ResolvedId resolvedId) {
    return new SecureProfile.Builder(secureProfile).resolvedId(resolvedId).build();
  }

}
