package com.kontakt.sdk.android.ble.connection;

import android.util.Base64;
import com.kontakt.sdk.android.ble.security.ResponseCode;
import com.kontakt.sdk.android.ble.security.exception.InvalidConfigException;
import com.kontakt.sdk.android.ble.security.parser.SimpleResponseParser;
import com.kontakt.sdk.android.common.model.Config;

import static com.kontakt.sdk.android.ble.connection.ErrorCause.INCORRECT_RESPONSE;
import static com.kontakt.sdk.android.ble.connection.SecureConfigMapper.fromReadResponse;

/**
 * Wrapper for {@link ReadListener} listener.
 * Parses and reports encrypted ReadAll command response to a {@link Config} object.
 */
class ReadAllListener implements ReadListener<Config> {

  private static final ReadListener<Config> NOOP_LISTENER = new ReadListener<Config>() {
    @Override
    public void onReadSuccess(Config value) {
      // No operation
    }

    @Override
    public void onReadFailure(ErrorCause cause) {
      //No operation
    }
  };

  private final String beaconPassword;
  private final ReadListener<Config> wrappedListener;

  public static ReadAllListener create(ReadListener<Config> readListener, String beaconPassword) {
    return new ReadAllListener(readListener, beaconPassword);
  }

  public static ReadAllListener noop() {
    return new ReadAllListener(NOOP_LISTENER, null);
  }

  private ReadAllListener(ReadListener<Config> readListener, String beaconPassword) {
    this.beaconPassword = beaconPassword;
    this.wrappedListener = readListener;
  }

  @Override
  public void onReadSuccess(Config config) {
    wrappedListener.onReadSuccess(config);
  }

  @Override
  public void onReadFailure(ErrorCause cause) {
    wrappedListener.onReadFailure(cause);
  }

  public void onResponseReceived(String response) {
    if (wrappedListener == NOOP_LISTENER) {
      return;
    }
    if (response == null || response.isEmpty()) {
      wrappedListener.onReadFailure(INCORRECT_RESPONSE);
      return;
    }

    if (!checkResponse(response)) {
      return;
    }

    Config parsedConfig = fromReadResponse(beaconPassword, response);
    if (parsedConfig != null) {
      wrappedListener.onReadSuccess(parsedConfig);
    } else {
      wrappedListener.onReadFailure(INCORRECT_RESPONSE);
    }
  }

  boolean checkResponse(String response) {
    byte[] decodedBase = Base64.decode(response, Base64.DEFAULT);
    try {
      SimpleResponseParser responseParser = SimpleResponseParser.of(decodedBase, beaconPassword);
      ResponseCode result = responseParser.getResult();
      switch (result) {
        case OK:
          return true;
        case VERIFICATION_FAILED:
          wrappedListener.onReadFailure(ErrorCause.VERIFICATION_FAILED);
          return false;
        case AUTHORIZATION_FAILED:
          wrappedListener.onReadFailure(ErrorCause.AUTHORIZATION_FAILED);
          return false;
        case ID_NOT_FOUND:
          wrappedListener.onReadFailure(ErrorCause.ID_NOT_FOUND);
          return false;
        default:
          wrappedListener.onReadFailure(ErrorCause.INCORRECT_RESPONSE);
          return false;
      }
    } catch (InvalidConfigException e) {
      wrappedListener.onReadFailure(ErrorCause.INCORRECT_RESPONSE);
      return false;
    }
  }
}
