package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.HashMap;
import java.util.Map;

/**
 * Model class that represents JSON structure of Kontakt.io activities's context.
 * To create new instance of this class, please use {@link ActivityContext.Builder}.
 */
public class ActivityContext implements Parcelable {

  public static final Creator<ActivityContext> CREATOR = new Creator<ActivityContext>() {
    @Override
    public ActivityContext createFromParcel(Parcel source) {
      return new ActivityContext(source);
    }

    @Override
    public ActivityContext[] newArray(int size) {
      return new ActivityContext[size];
    }
  };

  private final String url;
  private final HttpMethod httpMethod;
  private final Map<String, String> headers;
  private final Map<String, String> params;

  public static Builder builder() {
    return new Builder();
  }

  ActivityContext(Builder builder) {
    this.url = builder.url;
    this.httpMethod = builder.httpMethod;
    this.headers = builder.headers;
    this.params = builder.params;
  }

  protected ActivityContext(Parcel in) {
    this.url = in.readString();
    int tmpHttpMethod = in.readInt();
    this.httpMethod = tmpHttpMethod == -1 ? null : HttpMethod.values()[tmpHttpMethod];
    int headersSize = in.readInt();
    this.headers = new HashMap<>(headersSize);
    for (int i = 0; i < headersSize; i++) {
      String key = in.readString();
      String value = in.readString();
      this.headers.put(key, value);
    }
    int paramsSize = in.readInt();
    this.params = new HashMap<>(paramsSize);
    for (int i = 0; i < paramsSize; i++) {
      String key = in.readString();
      String value = in.readString();
      this.params.put(key, value);
    }
  }

  private ActivityContext() {
    this(new Builder());
  }

  public String getUrl() {
    return url;
  }

  public HttpMethod getHttpMethod() {
    return httpMethod;
  }

  public Map<String, String> getHeaders() {
    return headers;
  }

  public Map<String, String> getParameters() {
    return params;
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof ActivityContext)) {
      return false;
    }
    final ActivityContext activity = (ActivityContext) object;

    return SDKEqualsBuilder.start().equals(url, activity.url).equals(httpMethod, activity.httpMethod).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(url).append(httpMethod).build();
  }

  /**
   * Builder class that is used to build {@link ActivityContext} instances from values configured by the setters.
   */
  public static class Builder {

    String url;
    HttpMethod httpMethod;
    Map<String, String> headers;
    Map<String, String> params;

    public Builder url(final String url) {
      this.url = url;
      return this;
    }

    public Builder httpMethod(final HttpMethod method) {
      this.httpMethod = method;
      return this;
    }

    public Builder headers(final Map<String, String> headers) {
      this.headers = headers;
      return this;
    }

    public Builder parameters(final Map<String, String> params) {
      this.params = params;
      return this;
    }

    public ActivityContext build() {
      return new ActivityContext(this);
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.url);
    dest.writeInt(this.httpMethod == null ? -1 : this.httpMethod.ordinal());
    dest.writeInt(this.headers.size());
    for (Map.Entry<String, String> entry : this.headers.entrySet()) {
      dest.writeString(entry.getKey());
      dest.writeString(entry.getValue());
    }
    dest.writeInt(this.params.size());
    for (Map.Entry<String, String> entry : this.params.entrySet()) {
      dest.writeString(entry.getKey());
      dest.writeString(entry.getValue());
    }
  }
}
