package com.kontakt.sdk.android.ble.util;

import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.FirmwareVersion;
import com.kontakt.sdk.android.common.model.Model;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.util.EnumSet;

import static com.kontakt.sdk.android.common.model.Model.HEAVY_DUTY_BEACON;
import static com.kontakt.sdk.android.common.model.Model.SMART_BEACON_3;
import static com.kontakt.sdk.android.common.model.Model.TAG_BEACON;
import static com.kontakt.sdk.android.common.model.Model.GATEWAY;
import static com.kontakt.sdk.android.common.model.Model.BEACON_PRO;
import static com.kontakt.sdk.android.common.model.Model.PROBE;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public final class DeviceUtils {

  private static final EnumSet<Model> SECURE_PROFILES = EnumSet.of(
      GATEWAY, BEACON_PRO, PROBE, TAG_BEACON, SMART_BEACON_3, HEAVY_DUTY_BEACON
  );

  private DeviceUtils() {
    // prevent from creation
  }

  // Secure check methods

  public static boolean isSecure(final Device device) {
    checkNotNull(device, "Device cannot be null");
    return isSecure(device.getModel(), device.getFirmware());
  }

  public static boolean isSecure(final RemoteBluetoothDevice device) {
    checkNotNull(device, "Device cannot be null");
    return isSecure(device.getModel(), device.getFirmwareVersion());
  }

  public static boolean isSecure(final Model model, final String firmware) {
    checkNotNull(model, "Model cannot be null");
    checkNotNull(firmware, "Firmware version cannot be null");

    if (SECURE_PROFILES.contains(model)) {
      return true;
    }

    final FirmwareVersion minFirmwareVersion = new FirmwareVersion("4.0");
    final FirmwareVersion deviceFirmwareVersion = new FirmwareVersion(firmware);

    return deviceFirmwareVersion.isGreaterOrEqualTo(minFirmwareVersion);
  }

  // RTC & light sensor check methods

  public static boolean hasRtcFeature(final Device device) {
    checkNotNull(device, "Device cannot be null");
    return hasRtcFeature(device.getModel(), device.getFirmware());
  }

  public static boolean hasRtcFeature(final RemoteBluetoothDevice device) {
    checkNotNull(device, "Device cannot be null");
    return hasRtcFeature(device.getModel(), device.getFirmwareVersion());
  }

  public static boolean hasRtcFeature(final Model model, final String firmware) {
    checkNotNull(model, "Model cannot be null");
    checkNotNull(firmware, "Firmware version cannot be null");

    if (EnumSet.of(TAG_BEACON, SMART_BEACON_3, HEAVY_DUTY_BEACON).contains(model)) {
      return true;
    }

    final FirmwareVersion minFirmwareVersion = new FirmwareVersion("1.7");
    final FirmwareVersion deviceFirmwareVersion = new FirmwareVersion(firmware);

    return model == BEACON_PRO && deviceFirmwareVersion.isGreaterOrEqualTo(minFirmwareVersion);
  }

  public static boolean hasLightSensorFeature(final Device device) {
    checkNotNull(device, "Device cannot be null");
    return hasLightSensorFeature(device.getModel(), device.getFirmware());
  }

  public static boolean hasLightSensorFeature(final RemoteBluetoothDevice device) {
    checkNotNull(device, "Device cannot be null");
    return hasLightSensorFeature(device.getModel(), device.getFirmwareVersion());
  }

  public static boolean hasLightSensorFeature(final Model model, final String firmware) {
    checkNotNull(model, "Model cannot be null");
    checkNotNull(firmware, "Firmware version cannot be null");

    final FirmwareVersion minFirmwareVersion = new FirmwareVersion("1.7");
    final FirmwareVersion deviceFirmwareVersion = new FirmwareVersion(firmware);

    return model == BEACON_PRO && deviceFirmwareVersion.isGreaterOrEqualTo(minFirmwareVersion);
  }

  public static boolean hasRtcAndLightSensorFeatures(final Device device) {
    checkNotNull(device, "Device cannot be null");
    return hasRtcAndLightSensorFeatures(device.getModel(), device.getFirmware());
  }

  public static boolean hasRtcAndLightSensorFeatures(final RemoteBluetoothDevice device) {
    checkNotNull(device, "Device cannot be null");
    return hasRtcAndLightSensorFeatures(device.getModel(), device.getFirmwareVersion());
  }

  public static boolean hasRtcAndLightSensorFeatures(final Model model, final String firmware) {
    checkNotNull(model, "Model cannot be null");
    checkNotNull(firmware, "Firmware version cannot be null");

    return hasRtcFeature(model, firmware) && hasLightSensorFeature(model, firmware);
  }

  // Kontakt Telemetry broadcasting check methods

  public static boolean canTransmitKontaktTelemetry(final Device device) {
    checkNotNull(device, "Device cannot be null");
    return canTransmitKontaktTelemetry(device.getModel(), device.getFirmware());
  }

  public static boolean canTransmitKontaktTelemetry(final Model model, final String firmware) {
    checkNotNull(model, "Model cannot be null");
    checkNotNull(firmware, "Firmware version cannot be null");

    // Check if secure profile and not beacon pro
    if (model != BEACON_PRO && SECURE_PROFILES.contains(model)) {
      return true;
    }

    // If beacon pro then firmware revision must be checked additionally
    final FirmwareVersion minFirmwareVersion = new FirmwareVersion("1.10");
    final FirmwareVersion deviceFirmwareVersion = new FirmwareVersion(firmware);

    return model == BEACON_PRO && deviceFirmwareVersion.isGreaterOrEqualTo(minFirmwareVersion);
  }

  // Interleaving packets check methods

  public static boolean isInterleavingAvailable(final Device device) {
    checkNotNull(device, "Device cannot be null");
    return isInterleavingAvailable(device.getModel(), device.getFirmware());
  }

  public static boolean isInterleavingAvailable(final Model model, final String firmware) {
    checkNotNull(model, "Model cannot be null");
    checkNotNull(firmware, "Firmware version cannot be null");

    if (EnumSet.of(BEACON_PRO, TAG_BEACON, SMART_BEACON_3, HEAVY_DUTY_BEACON).contains(model)) {
      return true;
    }

    final FirmwareVersion minFirmwareVersion = new FirmwareVersion("4.1");
    final FirmwareVersion deviceFirmwareVersion = new FirmwareVersion(firmware);

    return deviceFirmwareVersion.isGreaterOrEqualTo(minFirmwareVersion);
  }

}
