package com.kontakt.sdk.android.ble.service;

import android.bluetooth.BluetoothDevice;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneDiscoverer;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconDiscoverer;
import com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileDiscoverer;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import static com.kontakt.sdk.android.common.profile.DeviceProfile.EDDYSTONE;
import static com.kontakt.sdk.android.common.profile.DeviceProfile.IBEACON;
import static com.kontakt.sdk.android.common.profile.DeviceProfile.KONTAKT_SECURE;

final class DefaultBluetoothDeviceDiscoverer implements BluetoothDeviceDiscoverer {

  final List<BluetoothDeviceDiscoverer> discoverers = new ArrayList<>();

  DefaultBluetoothDeviceDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract, FutureShufflesCache cache) {

    Set<IBeaconRegion> regions = scanContext.getIBeaconRegions();
    if (scanContext.getObservedProfiles().contains(IBEACON) || (!regions.isEmpty() && !regions.contains(BeaconRegion.EVERYWHERE))) {
      discoverers.add(new IBeaconDiscoverer(scanContext, discoveryContract, cache));
    }

    Set<IEddystoneNamespace> namespaces = scanContext.getEddystoneNamespaces();
    if (scanContext.getObservedProfiles().contains(EDDYSTONE) || (!namespaces.isEmpty() && !namespaces.contains(EddystoneNamespace.EVERYWHERE) )) {
      discoverers.add(new EddystoneDiscoverer(discoveryContract, scanContext, cache));
    }

    if (scanContext.getObservedProfiles().contains(KONTAKT_SECURE)) {
      discoverers.add(new SecureProfileDiscoverer(scanContext, discoveryContract, cache));
    }
  }

  @Override
  public void performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanResult) {
    for (BluetoothDeviceDiscoverer discoverer : discoverers) {
      discoverer.performDiscovery(bluetoothDevice, rssi, scanResult);
    }
  }

  @Override
  public void disable() {
    for (BluetoothDeviceDiscoverer discoverer : discoverers) {
      discoverer.disable();
    }
  }

  @Override
  public synchronized void evictInactiveDevices(long currentTimeMillis) {
    for (BluetoothDeviceDiscoverer discoverer : discoverers) {
      discoverer.evictInactiveDevices(currentTimeMillis);
    }
  }

}
