package com.kontakt.sdk.android.ble.discovery;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.device.BeaconDevice;
import com.kontakt.sdk.android.ble.device.EddystoneDevice;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SecureProfileUtils.asRemoteBluetoothDevice;

/**
 * This class is responsible for resolving any incoming shuffled device and passing the result further into discovery chain.
 */
public class ShuffledDevicesResolver {

  private final FutureShufflesCache cache;
  private final ResolveCallback callback;

  public ShuffledDevicesResolver(ResolveCallback resolveCallback, FutureShufflesCache cache) {
    this.callback = checkNotNull(resolveCallback);
    this.cache = checkNotNull(cache);
    this.cache.addCallback(resolveCallback);
  }

  public void resolve(ISecureProfile secureProfile) {
    resolve(asRemoteBluetoothDevice(secureProfile));
  }

  public void resolve(RemoteBluetoothDevice originalDevice) {
    if (originalDevice == null) {
      return;
    }
    if (!originalDevice.isShuffled()) {
      callback.onResolved(originalDevice);
      return;
    }

    ResolvedId resolvedId = cache.get(originalDevice);
    if (resolvedId == null) {
      cache.addResolveRequest(originalDevice);
      return;
    }
    if (FutureShufflesCache.PHANTOM_ENTRY.equals(resolvedId)) {
      return;
    }

    if (originalDevice.getUniqueId() == null) {
      RemoteBluetoothDevice resolvedDevice = createUpdatedDevice(originalDevice, resolvedId);
      callback.onResolved(resolvedDevice);
    } else {
      // if origin.uniqueId != null && origin.isShuffled
      // then the event was sent from resolvers
      // and can be directly further processed
      callback.onResolved(originalDevice);
    }
  }

  public void onDeviceLost(RemoteBluetoothDevice device) {
    ResolvedId resolvedId = cache.get(device);
    // not resolved yet
    if (resolvedId == null) {
      cache.markIgnored(device);
    }
  }

  public void disable() {
    cache.finishResolveRunners();
  }

  private static RemoteBluetoothDevice createUpdatedDevice(RemoteBluetoothDevice bluetoothDevice, ResolvedId resolvedId) {
    switch (bluetoothDevice.getProfile()) {
      case IBEACON:
        IBeaconDevice iBeaconDevice = (IBeaconDevice) bluetoothDevice;
        return BeaconDevice.of(iBeaconDevice, resolvedId);
      case EDDYSTONE:
        IEddystoneDevice eddystoneDevice = (IEddystoneDevice) bluetoothDevice;
        return new EddystoneDevice.Builder(eddystoneDevice).resolvedId(resolvedId).build();
      case KONTAKT_SECURE:
        ISecureProfile secureDevice = new SecureProfile.Builder(bluetoothDevice).resolvedId(resolvedId).build();
        return asRemoteBluetoothDevice(secureDevice);
      default:
        throw new IllegalArgumentException("Unsupported device profile!");
    }
  }

  public interface ResolveCallback {
    void onResolved(RemoteBluetoothDevice device);
  }
}

