package com.kontakt.sdk.android.common.util;

/**
 * The type Optional.
 *
 * @param <T> the type parameter
 *
 * A container object which may or may not contain a non-null value.
 * If a value is present, isPresent() will return true and get() will return the value.
 * Additional methods that depend on the presence or absence of a contained value are provided,
 * such as orElse() (return a default value if value not present) and ifPresent()
 * (execute a block of code if the value is present).
 */
public abstract class SDKOptional<T> {

  private static final Absent<Object> ABSENT_INSTANCE = new Absent<>();

  /**
   * Is present.
   *
   * @return the boolean
   */
  public abstract boolean isPresent();

  /**
   * Get t.
   *
   * @return the t
   */
  public abstract T get();

  /**
   * Of optional.
   *
   * @param <T> the type parameter
   * @param object the object
   * @return the optional
   */
  public static <T> SDKOptional<T> of(final T object) {
    SDKPreconditions.checkNotNull("Object is null.");
    return new Present<>(object);
  }

  /**
   * Absent optional.
   *
   * @param <T> the type parameter
   * @return the optional
   */
  @SuppressWarnings("unchecked")
  public static <T> SDKOptional<T> absent() {
    return (SDKOptional<T>) ABSENT_INSTANCE;
  }

  private static class Present<T> extends SDKOptional<T> {

    private final T object;

    /**
     * Instantiates a new Present.
     *
     * @param object the object
     */
    public Present(T object) {
      this.object = object;
    }

    @Override
    public boolean isPresent() {
      return true;
    }

    @Override
    public T get() {
      return object;
    }
  }

  private static class Absent<T> extends SDKOptional<T> {

    @Override
    public boolean isPresent() {
      return false;
    }

    @Override
    public T get() {
      throw new IllegalStateException("Object is absent.");
    }
  }
}
