package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import java.nio.ByteBuffer;
import java.util.Calendar;
import java.util.TimeZone;

import static com.kontakt.sdk.android.common.util.ConversionUtils.asInt;
import static com.kontakt.sdk.android.common.util.ConversionUtils.invert;
import static com.kontakt.sdk.android.common.util.ConversionUtils.to2ByteArray;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static java.util.Arrays.copyOfRange;

/**
 * Model class representing beacon's time.
 */
public class Time implements Parcelable {

  private static final int BLE_TIME_LENGTH = 10;

  public static final Creator<Time> CREATOR = new Creator<Time>() {
    @Override
    public Time createFromParcel(Parcel source) {
      return new Time(source);
    }

    @Override
    public Time[] newArray(int size) {
      return new Time[size];
    }
  };

  private final int year;
  private final int month;
  private final int dayOfMonth;
  private final int hour;
  private final int minute;
  private final int second;
  private final int dayOfWeek;

  public static Builder builder() {
    return new Builder();
  }

  /**
   * Returns Time instance parsed from BLE Current Time characteristic value.
   *
   * @param value {@link KontaktDeviceCharacteristic#CURRENT_TIME} value.
   * @return Time instance.
   */
  public static Time fromBleValue(byte[] value) {
    checkArgument(value.length == BLE_TIME_LENGTH, "Unexpected frame length. Should be: " + BLE_TIME_LENGTH);
    return new Time.Builder().year(asInt(invert(copyOfRange(value, 0, 2))))
        .month(value[2])
        .dayOfMonth(value[3])
        .hour(value[4])
        .minute(value[5])
        .second(value[6])
        .dayOfWeek(value[7])
        .build();
  }

  /**
   * Create Time instance based on current UTC time.
   *
   * @return Time instance.
   */
  public static Time getCurrentUTC() {
    Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
    int year = calendar.get(Calendar.YEAR);
    int month = calendar.get(Calendar.MONTH) + 1;
    int dayOfMonth = calendar.get(Calendar.DAY_OF_MONTH);
    int hour = calendar.get(Calendar.HOUR_OF_DAY);
    int minute = calendar.get(Calendar.MINUTE);
    int second = calendar.get(Calendar.SECOND);
    int calendarDayOfWeek = calendar.get(Calendar.DAY_OF_WEEK) - 1;
    int dayOfWeek = calendarDayOfWeek == 0 ? 7 : calendarDayOfWeek;
    return new Time.Builder()
        .year(year)
        .month(month)
        .dayOfMonth(dayOfMonth)
        .hour(hour)
        .minute(minute)
        .second(second)
        .dayOfWeek(dayOfWeek)
        .build();
  }

  protected Time(Parcel in) {
    this.hour = in.readInt();
    this.minute = in.readInt();
    this.second = in.readInt();
    this.dayOfWeek = in.readInt();
    this.dayOfMonth = in.readInt();
    this.month = in.readInt();
    this.year = in.readInt();
  }

  private Time() {
    this(new Builder());
  }

  Time(Builder builder) {
    this.hour = builder.hour;
    this.minute = builder.minute;
    this.second = builder.second;
    this.dayOfWeek = builder.dayOfWeek;
    this.dayOfMonth = builder.dayOfMonth;
    this.month = builder.month;
    this.year = builder.year;
  }

  /**
   * @return Current hour: 0 - 23
   */
  public int getHour() {
    return hour;
  }

  /**
   * @return Minutes: 0 - 59
   */
  public int getMinute() {
    return minute;
  }

  /**
   * @return Seconds: 0 - 59
   */
  public int getSecond() {
    return second;
  }

  /**
   * @return Day of week: Monday - 1, Sunday - 7
   */
  public int getDayOfWeek() {
    return dayOfWeek;
  }

  /**
   * @return Day of month: 1 - 31
   */
  public int getDayOfMonth() {
    return dayOfMonth;
  }

  /**
   * @return Month: January - 1, December - 12
   */
  public int getMonth() {
    return month;
  }

  /**
   * @return Year: 1582 - 9999
   */
  public int getYear() {
    return year;
  }

  /**
   * @return Time as byte array eligible for BLE characteristic write.
   */
  public byte[] toBleValue() {
    ByteBuffer buffer = ByteBuffer.allocate(BLE_TIME_LENGTH);
    buffer.put(to2ByteArray(year));
    buffer.put((byte) month);
    buffer.put((byte) dayOfMonth);
    buffer.put((byte) hour);
    buffer.put((byte) minute);
    buffer.put((byte) second);
    buffer.put((byte) dayOfWeek);
    return buffer.array();
  }

  @Override
  public String toString() {
    return "Time{" +
        "hour=" + hour +
        ", minute=" + minute +
        ", second=" + second +
        ", dayOfWeek=" + dayOfWeek +
        ", dayOfMonth=" + dayOfMonth +
        ", month=" + month +
        ", year=" + year +
        '}';
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    Time time = (Time) o;

    if (hour != time.hour) return false;
    if (minute != time.minute) return false;
    if (second != time.second) return false;
    if (dayOfWeek != time.dayOfWeek) return false;
    if (dayOfMonth != time.dayOfMonth) return false;
    if (month != time.month) return false;
    return year == time.year;
  }

  @Override
  public int hashCode() {
    int result = hour;
    result = 31 * result + minute;
    result = 31 * result + second;
    result = 31 * result + dayOfWeek;
    result = 31 * result + dayOfMonth;
    result = 31 * result + month;
    result = 31 * result + year;
    return result;
  }

  public static class Builder {

    int hour;
    int minute;
    int second;
    int dayOfWeek;
    int dayOfMonth;
    int month;
    int year;

    public Builder hour(int hour) {
      this.hour = hour;
      return this;
    }

    public Builder minute(int minute) {
      this.minute = minute;
      return this;
    }

    public Builder second(int second) {
      this.second = second;
      return this;
    }

    public Builder dayOfWeek(int dayOfWeek) {
      this.dayOfWeek = dayOfWeek;
      return this;
    }

    public Builder dayOfMonth(int dayOfMonth) {
      this.dayOfMonth = dayOfMonth;
      return this;
    }

    public Builder month(int month) {
      this.month = month;
      return this;
    }

    public Builder year(int year) {
      this.year = year;
      return this;
    }

    public Time build() {
      return new Time(this);
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.hour);
    dest.writeInt(this.minute);
    dest.writeInt(this.second);
    dest.writeInt(this.dayOfWeek);
    dest.writeInt(this.dayOfMonth);
    dest.writeInt(this.month);
    dest.writeInt(this.year);
  }
}
