package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * This class represents an Eddystone-UID composed of 10-byte namespace and a 6-byte instance ID.
 */
public class EddystoneUid implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<EddystoneUid> CREATOR = new Creator<EddystoneUid>() {
    @Override
    public EddystoneUid createFromParcel(Parcel source) {
      return new EddystoneUid(source);
    }

    @Override
    public EddystoneUid[] newArray(int size) {
      return new EddystoneUid[size];
    }
  };

  private final String namespace;
  private final String instanceId;

  /**
   * Creates new Eddystone-UID object with specified namespace and instance ID.
   *
   * @param namespace the namespace.
   * @param instanceId the instance ID.
   * @return the Eddystone-UID instance.
   */
  public static EddystoneUid of(String namespace, String instanceId) {
    return new EddystoneUid(namespace, instanceId);
  }

  /**
   * Creates new Eddystone-UID object from {@link IEddystoneDevice} instance.
   *
   * @param fakeDevice the Eddystone device instance.
   * @return the Eddystone-UID instance.
   */
  public static EddystoneUid fromDevice(IEddystoneDevice fakeDevice) {
    SDKPreconditions.checkNotNull(fakeDevice, "device cannot be null");
    return new EddystoneUid(fakeDevice.getNamespace(), fakeDevice.getInstanceId());
  }

  /**
   * Creates new Eddystone-UID object from String value.
   *
   * @param queriedBy the string value in namespace:instanceID format.
   * @return the Eddystone-UID instance.
   * @throws NullPointerException if input String is null.
   * @throws IllegalArgumentException if input String has invalid format.
   */
  public static EddystoneUid fromQueriedBy(String queriedBy) {
    SDKPreconditions.checkNotNull(queriedBy, "queriedBy is null");
    String[] split = queriedBy.split(":");
    if (split.length != 2) {
      throw new IllegalArgumentException("queriedBy has invalid format");
    }
    String namespace = split[0];
    String instanceId = split[1];
    return new EddystoneUid(namespace, instanceId);
  }

  private EddystoneUid(String namespace, String instanceId) {
    this.namespace = namespace;
    this.instanceId = instanceId;
  }

  protected EddystoneUid(Parcel in) {
    this.namespace = in.readString();
    this.instanceId = in.readString();
  }

  public String getNamespace() {
    return namespace;
  }

  public String getInstanceId() {
    return instanceId;
  }

  // do not change implementation of toString() method, it is used as http request param as well
  @Override
  public String toString() {
    return namespace + ":" + instanceId;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof EddystoneUid)) {
      return false;
    }
    if (o == this) {
      return true;
    }

    EddystoneUid eddystoneUid = (EddystoneUid) o;
    return SDKEqualsBuilder.start().equals(namespace, eddystoneUid.namespace).equals(instanceId, eddystoneUid.instanceId).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(namespace).append(instanceId).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.namespace);
    dest.writeString(this.instanceId);
  }
}
