package com.kontakt.sdk.android.cloud.api.executor.venues;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.VenuesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link VenuesApi}. Use this class if you want to unshare
 * venues through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.venues().unshare(IDs)
 *      .fromManagers(managerEmails)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that managers must be specified so invocation of {@code fromManagers} method is
 * mandatory. Otherwise an exception will be thrown at runtime.
 */
public class UnshareVenueRequestExecutor extends RequestExecutor<String> {

  private final VenuesService venuesService;

  private final UUID[] ids;
  private String[] managerEmails;

  /**
   * Constructs request executor initialized with corresponding service class and venue unique IDs.
   *
   * @param venuesService the venues API service.
   * @param ids venue unique identifiers.
   */
  public UnshareVenueRequestExecutor(final VenuesService venuesService, final UUID... ids) {
    this.venuesService = venuesService;
    this.ids = ids;
  }

  /**
   * Constructs request executor initialized with corresponding service class and venue unique IDs.
   *
   * @param venuesService the venues API service.
   * @param ids venue unique identifiers.
   */
  public UnshareVenueRequestExecutor(final VenuesService venuesService, final List<UUID> ids) {
    this.venuesService = venuesService;
    final int size = ids.size();
    this.ids = ids.toArray(new UUID[size]);
  }

  /**
   * Specifies managers. The method invocation is obligatory while using this request executor.
   *
   * @param managerEmails manager email addresses.
   * @return this request executor.
   */
  public UnshareVenueRequestExecutor fromManagers(final String... managerEmails) {
    SDKPreconditions.checkNotNull(managerEmails, "manager emails cannot be null");
    this.managerEmails = managerEmails;
    return this;
  }

  /**
   * Specifies managers. The method invocation is obligatory while using this request executor.
   *
   * @param managerEmails manager email addresses.
   * @return this request executor.
   */
  public UnshareVenueRequestExecutor fromManagers(final List<String> managerEmails) {
    SDKPreconditions.checkNotNull(managerEmails, "manager emails cannot be null");
    final int size = managerEmails.size();
    this.managerEmails = managerEmails.toArray(new String[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return venuesService.unshareVenue(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(managerEmails != null, "cannot unshare venue - specify managers");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Venues.VENUE_ID_PARAMETER, StringUtils.join(ids, ","));
    params.put("managerMail", StringUtils.join(managerEmails, ","));
    return params;
  }
}
