package com.kontakt.sdk.android.cloud.api.executor.receivers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ReceiversApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ReceiversService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.paginated.Receivers;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

/**
 * Request executor provided by {@link ReceiversApi}. Use this class if you want to obtain
 * receivers through fluent API in chained fashion, for instance:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   Receivers receivers = kontaktCloud.receivers().fetch()
 *      .fromPlaces(placeIDs)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class ReceiversRequestExecutor extends PaginatedRequestExecutor<Receivers> {

  private final ReceiversService receiversService;

  private UUID[] ids;
  private UUID[] placeIds;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param receiversService the receivers API service.
   */
  public ReceiversRequestExecutor(ReceiversService receiversService) {
    this.receiversService = receiversService;
  }

  /**
   * Specifies receivers identifiers. Use that method if you want to fetch particular receiver by ID.
   *
   * @param ids receivers identifiers.
   * @return this request executor.
   */
  public ReceiversRequestExecutor withIds(final UUID... ids) {
    this.ids = checkNotNull(ids, "Receiver IDs cannot be null");
    return this;
  }

  /**
   * Specifies receivers identifiers. Use that method if you want to fetch particular receiver by ID.
   *
   * @param ids receivers identifiers.
   * @return this request executor.
   */
  public ReceiversRequestExecutor withIds(final List<UUID> ids) {
    checkNotNull(ids, "Receiver IDs cannot be null");
    final int size = ids.size();
    this.ids = ids.toArray(new UUID[size]);
    return this;
  }

  /**
   * Specifies places for whose receivers belong to.
   *
   * @param placeIds places identifiers.
   * @return this request executor.
   */
  public ReceiversRequestExecutor fromPlaces(final UUID... placeIds) {
    this.placeIds = checkNotNull(placeIds, "Place IDs cannot be null");
    return this;
  }

  /**
   * Specifies places for whose receivers belong to.
   *
   * @param placeIds places identifiers.
   * @return this request executor.
   */
  public ReceiversRequestExecutor fromPlaces(final List<UUID> placeIds) {
    checkNotNull(placeIds, "Place IDs cannot be null");
    final int size = placeIds.size();
    this.placeIds = placeIds.toArray(new UUID[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ReceiversRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ReceiversRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ReceiversRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ReceiversRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ReceiversRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Receivers execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<Receivers> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Receivers> prepareCall() {
    final Call<Receivers> call;
    if (eTag != null) {
      call = receiversService.getReceivers(params(), eTag);
    } else {
      call = receiversService.getReceivers(params());
    }
    return call;
  }

  private void checkPreconditions() {
    checkState(ids != null || placeIds != null, "cannot fetch receivers - specify either receiver IDs or place IDs");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    if (ids != null) {
      params.put(CloudConstants.Receivers.RECEIVER_ID_PARAMETER, StringUtils.join(ids, ","));
    }
    if (placeIds != null) {
      params.put(CloudConstants.Receivers.PLACE_ID_PARAMETER, StringUtils.join(placeIds, ","));
    }
    return params;
  }
}
